import { Component, OnInit } from '@angular/core';
import { cloneDeep } from 'lodash';
import { shipmentModelInitializer } from 'src/app/ShipmentEntity/shipmentModelInitializer';
import { AddCustomer, AddShipment, AddWine, CreateShipment, Customer, CustomerListElement, Shipment, VehicleListElement, Wine, WineListElement } from 'src/app/ShipmentEntity/shipmentModelDefinition';
import { ShipmentService } from '../shipment.service';
import { ListElement } from 'src/app/models';
import { ActivatedRoute, Router } from '@angular/router';
import { firstValueFrom } from 'rxjs';
import { ShowErrorToaster, ShowSuccessToaster } from 'src/assets/ErrorHandler';
import { ToastrService } from 'ngx-toastr';
import { isAppInOfflineMode } from 'src/environments/app-settings';

@Component({
  selector: 'app-shipment-edit',
  templateUrl: './shipment-edit.component.html',
  styleUrls: ['./shipment-edit.component.scss']
})
export class ShipmentEditComponent implements OnInit {

  constructor(private shipmentService: ShipmentService, private router: Router, private route: ActivatedRoute, private toastr: ToastrService) {}

  shipment: Shipment = cloneDeep(shipmentModelInitializer.shipment);
  vehicleList: VehicleListElement[] = isAppInOfflineMode() ? [] : shipmentModelInitializer.vehicleList
  employeeList: ListElement[] = isAppInOfflineMode() ? [] :  shipmentModelInitializer.employeeList
  customerList: CustomerListElement[] = isAppInOfflineMode() ? [] :  shipmentModelInitializer.customerList
  wineList: WineListElement[] = isAppInOfflineMode() ? [] :  shipmentModelInitializer.wineList

  filteredVehicleList: VehicleListElement[]=[];

  vehicleCapacity: number = 0;
  currentVehicleCapacity: number = 0;

  disableCreateShipment: boolean = true;
  disableChoseVehicle: boolean = true;
  disableAddCustomer: boolean = true;
  disableAddWine: boolean = true;

  disableAddWineButton:boolean = true;
  disableAddCustomerButton: boolean = false

  shipmentId: number = 0;

  async ngOnInit(): Promise<void> {
    this.route.paramMap.subscribe(params => {
      this.shipmentId = Number(params.get('shipmentId'));
    });

    await this.fetchShipmentInfo()
    await this.getEditableShipment(this.shipmentId)
  }

  async getEditableShipment(shipmentId: number): Promise<void>{
    const editableShipment = await firstValueFrom(this.shipmentService.getEditableShipment(shipmentId))
      this.shipment = editableShipment

      this.shipment.customers.forEach((x: Customer)=>{
        x.disableAddWine = false;
        x.disableAddWineButton = false;
        x.wineList = this.wineList;
      })

      this.selectEmployee(this.shipment.employeeId)
      this.shouldCreateShipmentBeEnabled()
      this.disableAddCustomer = false
  }

  async fetchShipmentInfo(): Promise<void> {
    const data = await firstValueFrom(this.shipmentService.getCreateShipmentInfo())    
      this.employeeList = data.employeeList
      this.customerList = data.customerList
      this.wineList = data.wineList
      this.shipment.customers.every(c=>c.wineList = this.wineList)
      this.vehicleList = data.vehicleList
  }

  selectDate(dateValue: string){
    console.log(dateValue)
  }

  // #region Employee
  selectEmployee(employeeId: string){
    if(employeeId != ""){
      //filter vehicles by employee
      this.filteredVehicleList = this.vehicleList.filter(x=>x.employeeId == employeeId)
      this.disableChoseVehicle = false

      this.changeVehicle(this.shipment.vehicleId)
    }
  }
  // #endregion

  // #region Customer 
  addCustomer(){
    let wines: Wine[] = [{wineId:"", disableAddWineQuantity: true, quantity:0, disableAddWineAgreedPrice: true, agreedPrice:0, basePrice:0}]
    let customer: Customer = {customerId:"", disableAddWine: true, disableAddWineButton: true, wineList: this.wineList, wines:wines}
    this.shipment.customers.push(customer)
    this.shouldAddCustomerButtonBeEnabled();
  }

  selectCustomer(customerId: string){
    if(customerId != ""){
      this.shipment.customers.find(x=>x.customerId == customerId)!.disableAddWine = false
    }
  }

  removeCustomer(indexToRemove: number){
    this.shipment.customers.splice(indexToRemove, 1)
    this.shouldAddCustomerButtonBeEnabled();
    this.calculateWineQuantity();
  }

  checkIfCustomerIsAlreadyChosen(customerId: string): boolean{
    if(customerId != ""){
    return this.shipment.customers.some(x=>x.customerId == customerId)
    }
    return false
  }

  shouldAddCustomerButtonBeEnabled(){
    if(this.shipment.customers.length == this.customerList.length){
      this.disableAddCustomerButton = true;
    }
    else{
      this.disableAddCustomerButton = false;
    }
  }

  getCustomerAddress(customerId: string){
    return this.customerList.find(x=>x.id == customerId)?.address
  }
  // #endregion

  // #region Wine
  addWine(customer: Customer){
    let wine: Wine = {wineId:"", disableAddWineQuantity:true, quantity:0, disableAddWineAgreedPrice: true, agreedPrice:0, basePrice:0}
    this.disableCreateShipment = true;

    customer?.wines.push(wine)
    //check if add wine should be enabled per customer
    this.shouldAddWineButtonBeEnabled(customer);
  }

  selectWine(wine: Wine){
    wine.disableAddWineQuantity = false;
  }

  removeWine(customer: Customer, indexToRemove: number){
    customer?.wines.splice(indexToRemove, 1)
    //check if add wine should be enabled per customer
    this.shouldAddWineButtonBeEnabled(customer);
    this.calculateWineQuantity();
  }

  checkIfWineIsAlreadyChosen(wineId: string, customerId: string): boolean{
    if(wineId != "" && customerId != ""){
      return this.shipment.customers.find(x=>x.customerId == customerId)!.wines.some(x=>x.wineId == wineId)
    }
    return false
  }

  addWineQuantity(customer: Customer, wine: Wine){
    this.shouldCreateShipmentBeEnabled();
    //check if add wine should be enabled per customer
    this.shouldAddWineButtonBeEnabled(customer);

    if(wine.quantity >0) wine.disableAddWineAgreedPrice = false
    else wine.disableAddWineAgreedPrice = true
  }

  addWineAgreedPrice(customer: Customer){
    this.shouldCreateShipmentBeEnabled();
    this.shouldAddWineButtonBeEnabled(customer);
  }

  shouldAddWineButtonBeEnabled(customer: Customer){
    let customerWineLength = customer!.wines.length;

    if( customerWineLength == this.wineList.length){
      customer!.disableAddWineButton = true
    }
    else{
      let lastAddedWine = customer!.wines[customerWineLength - 1]
      if(lastAddedWine?.quantity <= 0 || lastAddedWine?.agreedPrice <= 0){
        customer!.disableAddWineButton = true
      }
      else {
        customer!.disableAddWineButton = false
      }
    }
  }

  getBasePricePerWine(wineId: string): number{
    return this.wineList.find(x=>x.id ==wineId)!.basePrice
  }

  calculateWinePricePercentageDifferecnce(basePrice: number, currentPrice: number): number{
    return ((currentPrice - basePrice) / basePrice) * 100
  }

  calculateWineQuantity(){
    let fullWineQuantityToShip: number = this.shipment.customers.flatMap(x=>x.wines).reduce((accumulator, wine) => accumulator + wine.quantity, 0);
    this.currentVehicleCapacity = fullWineQuantityToShip;
  }
  // #endregion

  // #region Vehicle
  selectVehicle(id: string){
    if(id != ""){
      this.vehicleCapacity = this.vehicleList.find(x=>x.id == id)!.capacity
      this.disableAddCustomer = false

      this.changeVehicle(id)
    }
  }

  changeVehicle(id: string = ""){
    if(this.currentVehicleCapacity == 0 && id == ""){
      this.shipment.vehicleId = ""
      this.vehicleCapacity = 0
    }
    else{
      let selectVehicle = id == "" ? this.filteredVehicleList.find(x=>x.capacity > this.currentVehicleCapacity)
                                   : this.filteredVehicleList.find(x=>x.capacity > this.currentVehicleCapacity && x.id == id);

      if(selectVehicle != undefined){
        this.shipment.vehicleId = selectVehicle.id
        this.vehicleCapacity = selectVehicle.capacity
      } 
      else{
        let winesQuantityToReset = this.shipment.customers.flatMap(x=>x.wines)
        winesQuantityToReset.forEach(wine => {
          wine.quantity = 0;
        });
        
        this.disableCreateShipment = true;
        this.shipment.vehicleId = ""
        this.vehicleCapacity = 0
        this.currentVehicleCapacity = 0
      }
    }
  }
  // #endregion

  // #region Shipment
  shouldCreateShipmentBeEnabled(){
    this.calculateWineQuantity();

    let doAllAddedWinesHaveQuantity = this.shipment.customers.flatMap(x=>x.wines).every(s=>s.quantity > 0)
    let doAllAddedWinesHaveAgreedPrice = this.shipment.customers.flatMap(x=>x.wines).every(s=>s.agreedPrice > 0)
    if((this.vehicleCapacity > this.currentVehicleCapacity) && this.currentVehicleCapacity > 0 && doAllAddedWinesHaveQuantity && doAllAddedWinesHaveAgreedPrice){
      this.disableCreateShipment = false
    }
    else{
      this.disableCreateShipment = true
    }
  }

  createShipment(){
    let editShipment: AddShipment = {
      shipmentId: this.shipment.shipmentId,
      shipmentDate: this.shipment.shipmentDate,
      employeeId: this.shipment.employeeId,
      vehicleId: this.shipment.vehicleId,
      customers: this.shipment.customers.map((s: AddCustomer) => ({
        customerId: s.customerId,
        wines: s.wines.map((w: AddWine) => ({
          wineId: w.wineId,
          quantity: w.quantity,
          agreedPrice: w.agreedPrice,
          basePrice: this.getBasePricePerWine(w.wineId)
      }))
    }))
  };

  this.shipmentService.editShipment(editShipment).subscribe(
    (isSuccessfull) => {
      this.router.navigate(['/shipment'])
      ShowSuccessToaster(this.toastr)
    },
    (response) => ShowErrorToaster(this.toastr, response)
  );
  }
}
// #endregion

