import { NextResponse } from 'next/server';
import prisma from 'src/lib/prisma';
import { tenantSchema } from 'src/schemas';
import { z } from 'zod';

export async function GET() {
  try {
    const tenants = await prisma.tenant.findMany();
    const tenant = tenants[0]; // Get first tenant since we're dealing with single tenant
    console.log('tenant', tenant);
    if (!tenant) {
      return NextResponse.json({ error: 'No tenant found' }, { status: 404 });
    }

    return NextResponse.json(tenant);
  } catch (error) {
    console.error('Error fetching tenant:', error);
    return NextResponse.json({ error: 'Failed to fetch tenant' }, { status: 500 });
  }
}

export async function POST(request: Request) {
  try {
    const body = await request.json();

    // Validate request body
    const validatedData = tenantSchema.parse(body);

    // Check if tenant already exists
    const existingTenants = await prisma.tenant.findMany();
    if (existingTenants.length > 0) {
      return NextResponse.json({ error: 'Tenant already exists' }, { status: 400 });
    }

    // Create new tenant
    const tenant = await prisma.tenant.create({
      data: validatedData,
    });

    return NextResponse.json(tenant, { status: 201 });
  } catch (error) {
    if (error instanceof z.ZodError) {
      return NextResponse.json(
        { error: 'Invalid request data', details: error.errors },
        { status: 400 }
      );
    }

    console.error('Error creating tenant:', error);
    return NextResponse.json({ error: 'Failed to create tenant' }, { status: 500 });
  }
}
