import { z } from 'zod';
import { customerSchema, tenantSchema } from './customer';

export const serviceSchema = z.object({
  id: z.string(),
  name: z.string(),
  sprint: z.number(),
  hour: z.number(),
  month: z.number(),
});

export const invoiceTableFilterValueSchema = z
  .union([z.string(), z.array(z.string()), z.date()])
  .nullable();

export const invoiceTableFiltersSchema = z.object({
  name: z.string(),
  service: z.array(z.string()),
  status: z.string(),
  startDate: z.date().nullable(),
  endDate: z.date().nullable(),
});

export const invoiceItemSchema = z.object({
  title: z.string(),
  price: z.number(),
  total: z.number(),
  service: serviceSchema,
  quantity: z.number(),
  description: z.string(),
});

export const invoiceStatusSchema = z.union([
  z.literal('draft'),
  z.literal('processing'),
  z.literal('pending'),
  z.literal('overdue'),
  z.literal('paid'),
]);

export const monthSchema = z.union([
  z.literal('January'),
  z.literal('February'),
  z.literal('March'),
  z.literal('April'),
  z.literal('May'),
  z.literal('June'),
  z.literal('July'),
  z.literal('August'),
  z.literal('September'),
  z.literal('October'),
  z.literal('November'),
  z.literal('December'),
]);

export const invoiceSchema = z.object({
  id: z.string(),
  sent: z.number().optional(),
  dueDate: z.coerce.date(),
  status: invoiceStatusSchema,
  currency: z.union([z.literal('EUR'), z.literal('USD')]),
  quantityType: z.union([
    z.literal('Unit'),
    z.literal('Hour'),
    z.literal('Sprint'),
    z.literal('Month'),
  ]),
  subTotal: z.number(),
  createDate: z.coerce.date(),
  month: monthSchema,
  discount: z.number().optional(),
  taxes: z.number().optional(),
  totalAmount: z.number(),
  invoiceNumber: z.string(),
  items: z.array(invoiceItemSchema),
  invoiceFrom: tenantSchema,
  invoiceTo: customerSchema,
  pdfRef: z.string().optional(),
});

export const updateInvoiceSchema = invoiceSchema.omit({
  id: true,
  sent: true,
});

export const createInvoiceSchema = invoiceSchema.omit({
  id: true,
  sent: true,
});

export interface InvoiceFilters {
  name: string;
  service: string[];
  status: string;
  startDate: Date | null;
  endDate: Date | null;
  where?: [string, string, any][];
}
