import { useMemo } from 'react';
// types
import { Employee } from 'src/schemas';
// db
import { endpoints, fetcher } from 'src/utils/axios';
import axios from 'src/utils/axios';
// swr
import useSWR, { mutate } from 'swr';

export function useGetEmployees() {
  const collectionName = endpoints.employee;

  const { data, isLoading, error, isValidating } = useSWR(collectionName, fetcher<Employee[]>, {
    revalidateOnFocus: false,
  });

  const memoizedValue = useMemo(
    () => ({
      employees: data || [],
      employeesLoading: isLoading,
      employeesError: error,
      employeesValidating: isValidating,
      employeesEmpty: !isLoading && !data?.length,
    }),
    [data, error, isLoading, isValidating]
  );

  return memoizedValue;
}

export async function createEmployee(employeeData: Partial<Employee>): Promise<Employee> {
  try {
    const response = await axios.post<Employee>(endpoints.employee, employeeData);

    // Mutate the SWR cache to include the new employee
    await mutate<Employee[]>(endpoints.employee, (currentData = []) => [
      ...currentData,
      response.data,
    ]);

    return response.data;
  } catch (error) {
    throw error;
  }
}
