import { NextRequest, NextResponse } from 'next/server';
import { customerTableFiltersSchema, newCustomerSchema } from 'src/schemas';
import prisma from 'src/lib/prisma';
import { authenticateRequest } from 'src/lib/auth-middleware';
import { CustomerStatus } from '@prisma/client';

export async function GET(request: NextRequest) {
  try {
    // Authenticate the request
    const authResult = await authenticateRequest(request);
    if (authResult instanceof NextResponse) {
      return authResult;
    }
    const { userId } = authResult;

    const searchParams = request.nextUrl.searchParams;
    const filters = {
      name: searchParams.get('name') || '',
      role: searchParams.getAll('role'),
      status: searchParams.get('status') || '',
    };

    // Validate filters
    const validatedFilters = customerTableFiltersSchema.parse(filters);

    const customers = await prisma.client.findMany({
      where: {
        name: { contains: validatedFilters.name, mode: 'insensitive' },
        status: validatedFilters.status
          ? { equals: validatedFilters.status as CustomerStatus }
          : undefined,
      },
    });
    console.log('customers', customers);

    return NextResponse.json(customers);
  } catch (error) {
    return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 });
  }
}

export async function POST(request: NextRequest) {
  try {
    // Authenticate the request
    const authResult = await authenticateRequest(request);
    if (authResult instanceof NextResponse) {
      return authResult;
    }
    const { userId } = authResult;

    const body = await request.json();
    const validatedData = newCustomerSchema.parse(body);
    console.log('validatedData', validatedData);

    const customer = await prisma.client.create({
      data: {
        ...validatedData,
        // userId,
      },
    });

    return NextResponse.json(customer, { status: 201 });
  } catch (error) {
    console.error('Error creating customer:', error);
    return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 });
  }
}
