import { NextRequest, NextResponse } from 'next/server';
import prisma from 'src/lib/prisma';
import { authenticateRequest } from 'src/lib/auth-middleware';

type InvoiceStatus = 'processing' | 'paid' | 'pending' | 'overdue' | 'draft';
type CurrencyTotals = { EUR: number; USD: number };
type Results = Record<InvoiceStatus | 'total', CurrencyTotals>;

export async function GET(request: NextRequest) {
  try {
    // Authenticate the request
    const authResult = await authenticateRequest(request);

    if (!authResult || authResult instanceof NextResponse) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const { userId, tenantId } = authResult;

    const searchParams = request.nextUrl.searchParams;
    const startDate = searchParams.get('startDate')
      ? new Date(searchParams.get('startDate')!)
      : null;

    if (!startDate) {
      return NextResponse.json({ error: 'Start date is required' }, { status: 400 });
    }

    // Calculate totals using raw SQL
    const totals = await prisma.$queryRaw`
      SELECT 
        status,
        currency,
        SUM("totalAmount") as total
      FROM "Invoice"
      WHERE "createDate" >= ${startDate}
        AND "invoiceFromId" = ${tenantId}
      GROUP BY status, currency
    `;

    // Format the response
    const results: Results = {
      total: { EUR: 0, USD: 0 },
      processing: { EUR: 0, USD: 0 },
      paid: { EUR: 0, USD: 0 },
      pending: { EUR: 0, USD: 0 },
      overdue: { EUR: 0, USD: 0 },
      draft: { EUR: 0, USD: 0 },
    };

    // Process the results - adjusted for raw SQL response format
    for (const row of totals as {
      status: string;
      currency: 'EUR' | 'USD';
      total: string | number;
    }[]) {
      const { status, currency, total } = row;
      const statusKey = status.toLowerCase() as InvoiceStatus;
      if (statusKey in results && (currency === 'EUR' || currency === 'USD')) {
        const amount = Number(total) || 0;
        results[statusKey][currency] = amount;
        results.total[currency] += amount;
      }
    }

    return NextResponse.json(results);
  } catch (error) {
    console.error('Error calculating totals:', error);
    return NextResponse.json({ error: 'Internal Server Error' }, { status: 500 });
  }
}
