import { useMemo } from 'react';
// types
import { Customer } from 'mvpmasters-shared';
// db
import { endpoints, fetcher } from 'src/utils/axios';
import axios from 'src/utils/axios';
// swr
import useSWR, { mutate } from 'swr';

export function useGetCustomers() {
  const collectionName = endpoints.customer;

  const { data, isLoading, error, isValidating } = useSWR(collectionName, fetcher<Customer[]>, {
    revalidateOnFocus: false,
  });

  const memoizedValue = useMemo(
    () => ({
      customers: data || [],
      customersLoading: isLoading,
      customersError: error,
      customersValidating: isValidating,
      customersEmpty: !isLoading && !data?.length,
    }),
    [data, error, isLoading, isValidating]
  );

  return memoizedValue;
}

export async function createCustomer(customerData: Partial<Customer>): Promise<Customer> {
  try {
    const response = await axios.post<Customer>(endpoints.customer, customerData);

    // Mutate the SWR cache to include the new customer
    await mutate<Customer[]>(endpoints.customer, (currentData = []) => [
      ...currentData,
      response.data,
    ]);

    return response.data;
  } catch (error) {
    throw error;
  }
}
