import { useMemo } from 'react';
import { Page, View, Text, Image, Document, Font, StyleSheet } from '@react-pdf/renderer';
// utils
import { fDate } from 'src/utils/format-time';
import { fCurrency } from 'src/utils/format-number';
// types
import { Invoice } from 'mvpmasters-shared';
import { createFullAddress } from 'src/utils/create-full-address';
import { getQuantityType } from 'src/utils/get-invoice-quantity-type';
// import signatureImage from 'src/assets/png/signature.png';

// ----------------------------------------------------------------------

Font.register({
  family: 'Roboto',
  fonts: [{ src: '/fonts/Roboto-Regular.ttf' }, { src: '/fonts/Roboto-Bold.ttf' }],
});

const useStyles = () =>
  useMemo(
    () =>
      StyleSheet.create({
        col4: { width: '25%' },
        col8: { width: '75%' },
        col6: { width: '50%' },
        mb4: { marginBottom: 4 },
        mb8: { marginBottom: 8 },
        mb40: { marginBottom: 40 },
        h3: { fontSize: 16, fontWeight: 700 },
        h4: { fontSize: 13, fontWeight: 700 },
        body1: { fontSize: 11 },
        body2: { fontSize: 10 },
        subtitle1: { fontSize: 11, fontWeight: 700 },
        subtitle2: { fontSize: 10, fontWeight: 700 },
        alignRight: { textAlign: 'right' },
        page: {
          fontSize: 9,
          lineHeight: 1.6,
          fontFamily: 'Roboto',
          backgroundColor: '#FFFFFF',
          textTransform: 'capitalize',
          padding: '40px 24px 120px 24px',
        },
        footer: {
          left: 0,
          right: 0,
          bottom: 0,
          padding: 24,
          margin: 'auto',
          borderTopWidth: 1,
          borderStyle: 'solid',
          position: 'absolute',
          borderColor: '#DFE3E8',
        },
        signatures: {
          left: 0,
          right: 0,
          bottom: 0,
          padding: 24,
          margin: 'auto',
          borderStyle: 'solid',
          position: 'absolute',
        },
        gridContainer: {
          flexDirection: 'row',
          justifyContent: 'space-between',
        },
        table: {
          display: 'flex',
          width: 'auto',
        },
        tableHead: {
          padding: '5px 5px',
          flexDirection: 'row',
        },
        tableRow: {
          padding: '5px 5px',
          flexDirection: 'row',
          borderBottomWidth: 1,
          borderStyle: 'solid',
          borderColor: '#DFE3E8',
        },
        noBorder: {
          paddingTop: 8,
          paddingBottom: 0,
          borderBottomWidth: 0,
        },
        tableCell_1: {
          width: '5%',
        },
        tableCell_2: {
          width: '50%',
          paddingRight: 16,
        },
        tableCell_3: {
          width: '15%',
        },
      }),
    []
  );

// ----------------------------------------------------------------------

type Props = {
  invoice: Invoice;
  currentStatus: string;
};

export default function InvoiceMKPDF({ invoice, currentStatus }: Props) {
  const {
    items,
    dueDate,
    discount,
    invoiceTo,
    createDate,
    totalAmount,
    invoiceFrom,
    invoiceNumber,
    subTotal,
    currency,
    quantityType,
  } = invoice;

  const styles = useStyles();

  return (
    <Document>
      <Page size="A4" style={styles.page}>
        <View style={[styles.gridContainer, styles.mb40]}>
          <Image source="/logo/logo_single.png" style={{ width: 48, height: 48 }} />

          <View style={{ alignItems: 'flex-end', flexDirection: 'column' }}>
            <Text style={styles.h3}>Invoice No. {invoiceNumber}</Text>
            <Text>{invoiceFrom.name}</Text>
            <Text>VAT Number: 4057020552553</Text>
            <Text>IBAN: MK07210701001535321</Text>
            <Text>SWIFT: TUTNMK22</Text>
          </View>
        </View>

        <View style={[styles.gridContainer, styles.mb40, { gap: 16 }]}>
          <View
            style={[
              styles.col6,
              { border: 1, borderColor: '#DFE3E8', borderRadius: 5, padding: 4 },
            ]}
          >
            <Text style={[styles.subtitle1, styles.mb4]}>Invoice from</Text>
            <Text style={[styles.body2, { fontWeight: 700 }]}>{invoiceFrom.name}</Text>
            <Text style={styles.body2}>{createFullAddress(invoiceFrom.address)}</Text>
            <Text style={styles.body2}>{invoiceFrom.phoneNumber}</Text>
            <View style={{ marginTop: 5 }} />
            <Text style={{ ...styles.body2, textTransform: 'lowercase' }}>{invoiceFrom.email}</Text>
          </View>

          <View
            style={[
              styles.col6,
              { border: 1, borderColor: '#DFE3E8', borderRadius: 5, padding: 4 },
            ]}
          >
            <Text style={[styles.subtitle1, styles.mb4]}>Invoice to</Text>
            <Text style={[styles.body2, { fontWeight: 700 }]}>{invoiceTo.name}</Text>
            {invoiceTo.companyId && (
              <Text style={[styles.body2]}>Company ID: {invoiceTo.companyId}</Text>
            )}
            <Text style={styles.body2}>{createFullAddress(invoiceTo.address)}</Text>
            <Text style={styles.body2}>{invoiceTo.phoneNumber}</Text>
            <View style={{ marginTop: 5 }} />
            <Text style={{ ...styles.body2, textTransform: 'lowercase' }}>{invoiceTo.email}</Text>
          </View>
        </View>

        <View style={[styles.gridContainer, styles.mb40]}>
          <View style={styles.col6}>
            <Text style={[styles.subtitle1, styles.mb4]}>Invoice Number</Text>
            <Text style={styles.body2}>{invoiceNumber}</Text>
          </View>
          <View style={styles.col6}>
            <Text style={[styles.subtitle1, styles.mb4]}>Date Issued</Text>
            <Text style={styles.body2}>{fDate(createDate.toDate())}</Text>
          </View>
          <View style={styles.col6}>
            <Text style={[styles.subtitle1, styles.mb4]}>Due date</Text>
            <Text style={styles.body2}>{fDate(dueDate.toDate())}</Text>
          </View>
        </View>

        <Text style={[styles.subtitle1, styles.mb8]}>Invoice Details</Text>

        <View style={styles.table}>
          <View>
            <View style={[styles.tableHead, { backgroundColor: 'black', color: 'white' }]}>
              <View style={styles.tableCell_1}>
                <Text style={styles.subtitle1}>#</Text>
              </View>

              <View style={styles.tableCell_2}>
                <Text style={styles.subtitle1}>Description</Text>
              </View>

              <View style={styles.tableCell_3}>
                <Text style={styles.subtitle1}>{getQuantityType(invoice)}</Text>
              </View>

              <View style={styles.tableCell_3}>
                <Text style={styles.subtitle1}>Unit price</Text>
              </View>

              <View style={[styles.tableCell_3, styles.alignRight]}>
                <Text style={styles.subtitle1}>Total</Text>
              </View>
            </View>
          </View>

          <View>
            {items.map((item, index) => (
              <View style={styles.tableRow} key={item.title}>
                <View style={styles.tableCell_1}>
                  <Text>{index + 1}</Text>
                </View>

                <View style={styles.tableCell_2}>
                  <Text style={styles.subtitle2}>{item.title}</Text>
                  <Text>{item.description}</Text>
                </View>

                <View style={styles.tableCell_3}>
                  <Text>{item.quantity}</Text>
                </View>

                <View style={styles.tableCell_3}>
                  <Text>{item.price}</Text>
                </View>

                <View style={[styles.tableCell_3, styles.alignRight]}>
                  <Text>{fCurrency(item.price * item.quantity, currency)}</Text>
                </View>
              </View>
            ))}

            <View style={[styles.tableRow, styles.noBorder]}>
              <View style={styles.tableCell_1} />
              <View style={styles.tableCell_2} />
              <View style={styles.tableCell_3} />
              <View style={styles.tableCell_3}>
                <Text>Subtotal</Text>
              </View>
              <View style={[styles.tableCell_3, styles.alignRight]}>
                <Text>{fCurrency(subTotal, currency)}</Text>
              </View>
            </View>

            {!!discount && (
              <View style={[styles.tableRow, styles.noBorder]}>
                <View style={styles.tableCell_1} />
                <View style={styles.tableCell_2} />
                <View style={styles.tableCell_3} />
                <View style={styles.tableCell_3}>
                  <Text>Discount</Text>
                </View>
                <View style={[styles.tableCell_3, styles.alignRight]}>
                  <Text>{fCurrency(-discount, currency)}</Text>
                </View>
              </View>
            )}

            {/* <View style={[styles.tableRow, styles.noBorder]}>
              <View style={styles.tableCell_1} />
              <View style={styles.tableCell_2} />
              <View style={styles.tableCell_3} />
              <View style={styles.tableCell_3}>
                <Text>Taxes</Text>
              </View>
              <View style={[styles.tableCell_3, styles.alignRight]}>
                <Text>{fCurrency(taxes)}</Text>
              </View>
            </View> */}

            <View style={[styles.tableRow, styles.noBorder]}>
              <View style={styles.tableCell_1} />
              <View style={styles.tableCell_2} />
              <View style={styles.tableCell_3} />
              <View style={styles.tableCell_3}>
                <Text style={styles.h4}>Total</Text>
              </View>
              <View style={[styles.tableCell_3, styles.alignRight]}>
                <Text style={styles.h4}>{fCurrency(totalAmount, currency)}</Text>
              </View>
            </View>
          </View>
        </View>

        {/* Add this block for signature lines */}
        <View style={[styles.gridContainer, styles.signatures, { marginBottom: 75 }]} fixed>
          <View style={[styles.gridContainer, { justifyContent: 'space-between' }]}>
            <View style={{ alignItems: 'flex-start', width: '50%' }}>
              {/* <Image
                source={signatureImage.src}
                style={{ width: 100, position: 'absolute', bottom: 20 }}
              /> */}
              <View
                style={{
                  marginTop: 4,
                  borderTopWidth: 1,
                  borderStyle: 'solid',
                  borderColor: '#000',
                  width: '65%',
                }}
              />
              <Text style={{ marginTop: 16, fontSize: 11 }}>{invoiceFrom.representative}</Text>
              <Text style={{ fontSize: 11, fontWeight: 700 }}>{invoiceFrom.name}</Text>
            </View>
            <View style={{ alignItems: 'flex-end', width: '50%' }}>
              <View
                style={{
                  marginTop: 4,
                  borderTopWidth: 1,
                  borderStyle: 'solid',
                  borderColor: '#000',
                  width: '65%',
                }}
              />
              <Text style={{ marginTop: 16, fontSize: 11 }}>{invoiceTo.representative}</Text>
              <Text style={{ fontSize: 11, fontWeight: 700 }}>{invoiceTo.name}</Text>
            </View>
          </View>
        </View>

        <View style={[styles.gridContainer, styles.footer]} fixed>
          <View style={styles.col8}>
            <Text style={styles.subtitle2}>NOTES</Text>
            <Text>
              VAT is not calculated according to article 14, paragraph 3, point 5 from the VAT act.
            </Text>
          </View>
          <View style={[styles.col4, styles.alignRight]}>
            <Text style={[styles.subtitle2]}>Have a question?</Text>
            <Text style={{ textTransform: 'lowercase' }}>{invoiceFrom.email}</Text>
          </View>
        </View>
      </Page>
    </Document>
  );
}
