import { useMemo } from 'react';
// types
import { Invoice, UpdateInvoice } from 'src/schemas';
// db
import useSWR from 'swr';
import { endpoints, fetcher } from 'src/utils/axios';
import axios from 'src/utils/axios';
import { mutate } from 'swr';
import { useSWRConfig } from 'swr';

interface InvoiceFilters {
  name?: string;
  service?: string[];
  status?: string;
  startDate?: string | null;
  endDate?: string | null;
}

export function useGetInvoices(params: InvoiceFilters = {}) {
  const collectionName = endpoints.invoice;

  const searchParams = new URLSearchParams();
  if (params.name) searchParams.set('name', params.name);
  if (params.status) searchParams.set('status', params.status);
  if (params.startDate) searchParams.set('startDate', params.startDate);
  if (params.endDate) searchParams.set('endDate', params.endDate);
  if (params.service?.length) {
    params.service.forEach((service) => searchParams.append('service', service));
  }

  const queryString = searchParams.toString();
  const endpoint = queryString ? `${collectionName}?${queryString}` : collectionName;

  const { data, isLoading, error, isValidating } = useSWR(
    endpoint,
    () => fetcher<Invoice[]>(endpoint),
    {
      revalidateOnFocus: false,
    }
  );

  const memoizedValue = useMemo(
    () => ({
      invoices: data || [],
      invoicesLoading: isLoading,
      invoicesError: error,
      invoicesValidating: isValidating,
      invoicesEmpty: !isLoading && !data?.length,
    }),
    [data, error, isLoading, isValidating]
  );

  return memoizedValue;
}

export function useGetInvoice({ id }: { id: string }) {
  const path = endpoints.invoice;

  const { data, isLoading, error, isValidating } = useSWR(
    `${path}/${id}`,
    () => fetcher<Invoice>(`${path}/${id}`),
    {
      revalidateOnFocus: false,
    }
  );

  const memoizedValue = useMemo(
    () => ({
      currentInvoice: data || null,
      currentInvoiceLoading: isLoading,
      currentInvoiceError: error,
      currentInvoiceValidating: isValidating,
      currentInvoiceEmpty: !isLoading && !data,
    }),
    [data, error, isLoading, isValidating]
  );

  return memoizedValue;
}

// Add this interface for the create invoice payload
interface CreateInvoicePayload {
  issueDate: Date;
  dueDate: Date;
  items: any[];
  invoiceNumber: string;
  invoiceFrom: any;
  invoiceTo: any;
  currency: string;
  quantityType: string;
  month: string;
  status?: string;
  taxes?: number;
  discount?: number;
  totalAmount: number;
  pdfRef?: string;
}

export async function createInvoice(data: CreateInvoicePayload): Promise<Invoice> {
  const response = await axios.post<Invoice>(endpoints.invoice, data);

  // Mutate the SWR cache to include the new invoice
  await mutate(
    endpoints.invoice,
    (existingInvoices: Invoice[] = []) => [response.data, ...existingInvoices],
    false // Set to false to avoid revalidation since we already have the new data
  );

  return response.data;
}

export async function updateInvoice(id: string, data: Partial<UpdateInvoice>) {
  const response = await axios.patch<Invoice>(`${endpoints.invoice}/${id}`, data);

  // Mutate the individual invoice cache
  await mutate(`${endpoints.invoice}/${id}`, response.data, false);

  // Mutate the invoice list cache
  await mutate(
    endpoints.invoice,
    (existingInvoices: Invoice[] = []) =>
      existingInvoices.map((invoice) => (invoice.id === id ? response.data : invoice)),
    false
  );

  return response.data;
}

export async function deleteInvoice(id: string) {
  const response = await axios.delete<Invoice>(`${endpoints.invoice}/${id}`);

  // Mutate the invoice list cache to remove the deleted invoice
  await mutate(
    endpoints.invoice,
    (existingInvoices: Invoice[] = []) => existingInvoices.filter((invoice) => invoice.id !== id),
    false
  );

  return response.data;
}

// Update the useDeleteInvoice hook to use the new implementation
export function useDeleteInvoice() {
  const deleteInvoiceMutation = async (id: string) => {
    try {
      await deleteInvoice(id);
      return true;
    } catch (error) {
      console.error('Error deleting invoice:', error);
      throw error;
    }
  };

  return { deleteInvoiceMutation };
}
