import { PrismaClient } from '@prisma/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

async function applyAdvancedSQLFeatures(prisma) {
  console.log('🔧 Applying Advanced SQL Features...');

  try {
    // Apply Complex Invoice Status Trigger
    console.log('📊 Applying Invoice Status Trigger...');
    const triggerSQL = fs.readFileSync(
      path.join(process.cwd(), 'sql/01_invoice_status_trigger.sql'),
      'utf8'
    );
    
    await prisma.$executeRawUnsafe(triggerSQL);
    console.log('✅ Invoice status trigger applied successfully!');

    // Apply Client Email Index Optimization
    console.log('📈 Applying Client Email Index Optimization...');
    const indexSQL = fs.readFileSync(
      path.join(process.cwd(), 'sql/02_client_email_index.sql'),
      'utf8'
    );
    
    await prisma.$executeRawUnsafe(indexSQL);
    console.log('✅ Email index optimization applied successfully!');

    // Verify implementations
    const triggerCheck = await prisma.$queryRaw`
      SELECT tgname FROM pg_trigger WHERE tgname = 'invoice_status_update_trigger'
    `;
    
    const indexCheck = await prisma.$queryRaw`
      SELECT count(*) as index_count
      FROM pg_indexes 
      WHERE tablename = 'Client' AND indexname LIKE 'idx_client_email%'
    `;

    console.log(`✅ Trigger verification: ${triggerCheck.length > 0 ? 'Active' : 'Not found'}`);
    console.log(`✅ Index verification: ${indexCheck[0].index_count} email indexes created`);

    return true;
  } catch (error) {
    console.error('❌ Error applying advanced SQL features:', error.message);
    
    // Don't fail the entire seeding process for SQL features
    // This allows the basic seeding to complete even if SQL features fail
    console.log('⚠️  Continuing with basic seeding...');
    return false;
  }
}

async function main() {
  // Clear existing data
  await prisma.service.deleteMany();
  await prisma.user.deleteMany();
  await prisma.tenant.deleteMany();

  // Define default tenant data
  const tenantData = {
    id: "cm7lxc3p00000pb7kmdrxsfod",
    name: "MVP Masters",
    email: "info@mvpmasters.com",
    address: {
      street: "Makedonska Treta Brigada 56",
      city: "Skopje",
      state: "Macedonia",
      zip: "1000",
      country: "Macedonia"
    },
    phoneNumber: "+389 72 233 943",
    representative: "Naum Shapkarovski",
    lastInvoiceNumber: "1",
    logoUrl: "https://example.com/default-logo.png",
    vatNumber: "VAT123456789",
    companyNumber: "COMP123456",
    bankAccounts: {
      eur: {
        accountNumber: "1234567890",
        routingNumber: "987654321",
        bicSwift: "DEFBANKXXX",
        iban: "DE89370400440532013000"
      },
      usd: {
        accountNumber: "0987654321",
        routingNumber: "123456789",
        bicSwift: "DEFBANKXXX",
        iban: "US89370400440532013000"
      }
    },
    // Add services along with the tenant creation
    services: {
      create: [
        {
          name: "Web Development",
          sprint: 5000,
          hour: 150,
          month: 8000
        },
        {
          name: "UI/UX Design",
          sprint: 3000,
          hour: 120,
          month: 6000
        },
        {
          name: "Consulting",
          sprint: 4000,
          hour: 200,
          month: 7000
        }
      ]
    }
  };

  // Create default tenant with services
  const defaultTenant = await prisma.tenant.create({
    data: tenantData,
    include: {
      services: true
    }
  });

  // Add default admin user
  const defaultUser = await prisma.user.create({
    data: {
      id: 'dK2y3WezYWWltHCaBRvUFlddkOr2',
      uid: 'dK2y3WezYWWltHCaBRvUFlddkOr2',
      email: 'naum@mvpmasters.com',
      displayName: 'Naum',
      role: 'ADMIN',
      tenantId: 'cm7lxc3p00000pb7kmdrxsfod'
    }
  });

  console.log('🌱 Seeding database...');
  console.log('Seeded default tenant:', defaultTenant);
  console.log('Seeded default user:', defaultUser);

  // Apply advanced SQL features after basic seeding
  const sqlFeaturesApplied = await applyAdvancedSQLFeatures(prisma);

  console.log('✅ Seeding complete!');
  
  if (sqlFeaturesApplied) {
    console.log('🎉 Advanced SQL features also applied!');
    console.log('   • Complex invoice status trigger with business logic');
    console.log('   • Optimized email indexes for Client table');
  } else {
    console.log('ℹ️  Advanced SQL features skipped (check sql/ directory exists)');
  }
}

main()
  .catch((e) => {
    console.error('Error seeding database:', e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
