-- Index Optimization for Client Email Queries
-- This creates an optimized index for queries like: SELECT * FROM "Client" WHERE email = 'test@example.com'

-- Drop existing index if it exists (Prisma may have created a basic one)
DROP INDEX IF EXISTS idx_client_email_basic;

-- Create a B-tree index for exact email matches
-- B-tree is optimal for equality queries and is the default index type in PostgreSQL
CREATE INDEX IF NOT EXISTS idx_client_email_btree ON "Client"(email);

-- Create a partial index for active clients only (most common use case)
-- This is smaller and faster for queries filtering active clients by email
CREATE INDEX IF NOT EXISTS idx_client_email_active ON "Client"(email) 
WHERE status = 'ACTIVE';

-- Create a composite index for email + tenant_id (multi-tenant optimization)
-- This covers queries that filter by both email and tenant
CREATE INDEX IF NOT EXISTS idx_client_email_tenant ON "Client"(email, tenant_id);

-- Create a case-insensitive index for email searches
-- This allows case-insensitive email lookups using functional index
CREATE INDEX IF NOT EXISTS idx_client_email_lower ON "Client"(LOWER(email));

-- Performance analysis queries (for testing the indexes)
-- You can run these to verify the indexes are being used:

/*
-- Test exact email match (should use idx_client_email_btree)
EXPLAIN (ANALYZE, BUFFERS) 
SELECT * FROM "Client" WHERE email = 'test@example.com';

-- Test case-insensitive email match (should use idx_client_email_lower)
EXPLAIN (ANALYZE, BUFFERS) 
SELECT * FROM "Client" WHERE LOWER(email) = LOWER('Test@Example.com');

-- Test email + tenant lookup (should use idx_client_email_tenant)
EXPLAIN (ANALYZE, BUFFERS) 
SELECT * FROM "Client" WHERE email = 'test@example.com' AND tenant_id = 'some-tenant-id';

-- Test active client email lookup (should use idx_client_email_active)
EXPLAIN (ANALYZE, BUFFERS) 
SELECT * FROM "Client" WHERE email = 'test@example.com' AND status = 'ACTIVE';
*/

-- Update table statistics for query planner optimization
ANALYZE "Client"; 