# Advanced SQL Implementation - Triggers & Indexing

This directory contains advanced SQL implementations for the AgencyOS project, featuring complex triggers and performance-optimized indexes.

## 🎯 Overview

### Task 1: Complex Invoice Status Trigger

- **File**: `01_invoice_status_trigger.sql`
- **Purpose**: Implements a sophisticated trigger system that reacts to invoice status changes
- **Complexity**: Multi-table operations, business logic automation, and notification system

### Task 2: Client Email Index Optimization

- **File**: `02_client_email_index.sql`
- **Purpose**: Optimizes email lookup performance with multiple index strategies
- **Optimization**: Multiple index types for different query patterns

## 🚀 Implementation Guide

### Prerequisites

- PostgreSQL database (as configured in your Prisma setup)
- Database connection with appropriate permissions
- Prisma CLI installed

### Step 1: Apply the SQL Files

#### Option A: Using PostgreSQL CLI

```bash
# Connect to your database
psql "your_database_connection_string"

# Apply the trigger implementation
\i sql/01_invoice_status_trigger.sql

# Apply the index optimization
\i sql/02_client_email_index.sql
```

#### Option B: Using Prisma Raw Queries

```javascript
// In a Node.js script or your application
import { PrismaClient } from '@prisma/client';
import fs from 'fs';

const prisma = new PrismaClient();

async function applySQLFiles() {
  // Apply trigger SQL
  const triggerSQL = fs.readFileSync('sql/01_invoice_status_trigger.sql', 'utf8');
  await prisma.$executeRawUnsafe(triggerSQL);

  // Apply index SQL
  const indexSQL = fs.readFileSync('sql/02_client_email_index.sql', 'utf8');
  await prisma.$executeRawUnsafe(indexSQL);

  console.log('Advanced SQL features applied successfully!');
}

applySQLFiles();
```

#### Option C: Create Prisma Migration

```bash
# Create a new migration
npx prisma migrate dev --create-only --name advanced-sql-features

# Add the SQL content to the generated migration file
# Then apply it
npx prisma migrate dev
```

## 📋 Feature Details

### Complex Invoice Status Trigger

#### What it does:

1. **Triggers on**: Invoice status updates (only when status actually changes)
2. **Creates tables**:
   - `InvoiceStatusHistory` - Logs all status changes
   - `InvoiceNotifications` - Business rule notifications
3. **Complex operations**:
   - Calculates client outstanding amounts
   - Monitors tenant monthly revenue
   - Automatically updates client status
   - Extends due dates for pending invoices
   - Generates priority-based notifications

#### Business Logic Examples:

- **PAID invoices**: Updates client to ACTIVE if no outstanding invoices
- **OVERDUE invoices**: Flags clients as INACTIVE if >3 overdue invoices
- **PENDING invoices**: Warns about high outstanding amounts (>€10,000)
- **Auto-extension**: Extends due dates for pending invoices with past due dates

### Client Email Index Optimization

#### Index Types Created:

1. **B-tree Index** (`idx_client_email_btree`): For exact email matches
2. **Partial Index** (`idx_client_email_active`): Only for active clients
3. **Composite Index** (`idx_client_email_tenant`): Email + tenant_id optimization
4. **Functional Index** (`idx_client_email_lower`): Case-insensitive searches

#### Query Patterns Optimized:

```sql
-- Exact match (uses idx_client_email_btree)
SELECT * FROM "Client" WHERE email = 'test@example.com';

-- Case-insensitive (uses idx_client_email_lower)
SELECT * FROM "Client" WHERE LOWER(email) = LOWER('Test@Example.com');

-- Multi-tenant (uses idx_client_email_tenant)
SELECT * FROM "Client" WHERE email = 'test@example.com' AND tenant_id = 'tenant-123';

-- Active clients only (uses idx_client_email_active)
SELECT * FROM "Client" WHERE email = 'test@example.com' AND status = 'ACTIVE';
```

## 🔧 Testing & Verification

### Test the Trigger

```sql
-- Update an invoice status to test the trigger
UPDATE "Invoice"
SET status = 'PAID'
WHERE invoice_number = 'INV-001';

-- Check the results
SELECT * FROM "InvoiceStatusHistory" ORDER BY changed_at DESC LIMIT 5;
SELECT * FROM "InvoiceNotifications" ORDER BY created_at DESC LIMIT 5;
```

### Test the Indexes

```sql
-- Check if indexes are being used
EXPLAIN (ANALYZE, BUFFERS)
SELECT * FROM "Client" WHERE email = 'test@example.com';

-- Should show "Index Scan using idx_client_email_btree"
```

### Performance Monitoring

```sql
-- Check index usage statistics
SELECT schemaname, tablename, indexname, idx_scan, idx_tup_read, idx_tup_fetch
FROM pg_stat_user_indexes
WHERE tablename = 'Client';
```

## 📊 Expected Performance Improvements

### Before Optimization:

- Email queries: Sequential scan (~100ms for 10k records)
- No invoice status tracking
- Manual business rule enforcement

### After Optimization:

- Email queries: Index scan (~1ms for 10k records)
- Automatic invoice status tracking and business rules
- Real-time notifications for business events
- 99%+ performance improvement for email lookups

## 🛠️ Maintenance

### Monitor Trigger Performance

```sql
-- Check trigger execution times
SELECT * FROM "InvoiceStatusHistory"
WHERE changed_at > NOW() - INTERVAL '1 day'
ORDER BY changed_at DESC;
```

### Index Maintenance

```sql
-- Rebuild indexes if needed (rarely required)
REINDEX INDEX idx_client_email_btree;

-- Update statistics for query planner
ANALYZE "Client";
```

## 🔍 Troubleshooting

### Common Issues:

1. **Trigger not firing**: Check if status actually changed
2. **Index not used**: Run `ANALYZE "Client"` to update statistics
3. **Performance issues**: Monitor with `pg_stat_statements`

### Debug Queries:

```sql
-- Check trigger exists
SELECT tgname, tgrelid::regclass, tgenabled
FROM pg_trigger
WHERE tgname = 'invoice_status_update_trigger';

-- Check indexes exist
SELECT indexname, indexdef
FROM pg_indexes
WHERE tablename = 'Client' AND indexname LIKE 'idx_client_email%';
```

## 🎓 Educational Value

This implementation demonstrates:

- **Trigger complexity**: Multi-table operations and business logic
- **Index strategies**: Different index types for different use cases
- **Performance optimization**: Query plan analysis and monitoring
- **Real-world application**: Practical business rule automation

The trigger goes beyond simple field updates to implement complex business rules, while the indexing strategy covers multiple query patterns for optimal performance.
