#!/usr/bin/env node

/**
 * Advanced SQL Features Application Script
 * 
 * This script applies the complex trigger and indexing optimizations
 * to your PostgreSQL database using Prisma's raw query capabilities.
 * 
 * Usage:
 *   node sql/apply-advanced-sql.js
 * 
 * Or add to package.json scripts:
 *   "apply-advanced-sql": "node sql/apply-advanced-sql.js"
 */

const { PrismaClient } = require('@prisma/client');
const fs = require('fs');
const path = require('path');

const prisma = new PrismaClient();

async function applyAdvancedSQL() {
  console.log('🚀 Applying Advanced SQL Features...\n');

  try {
    // Step 1: Apply Complex Invoice Status Trigger
    console.log('📊 Step 1: Applying Complex Invoice Status Trigger...');
    const triggerSQL = fs.readFileSync(
      path.join(__dirname, '01_invoice_status_trigger.sql'),
      'utf8'
    );
    
    await prisma.$executeRawUnsafe(triggerSQL);
    console.log('✅ Invoice status trigger applied successfully!');
    
    // Verify trigger creation
    const triggerCheck = await prisma.$queryRaw`
      SELECT tgname, tgrelid::regclass as table_name, tgenabled 
      FROM pg_trigger 
      WHERE tgname = 'invoice_status_update_trigger'
    `;
    
    if (triggerCheck.length > 0) {
      console.log('✅ Trigger verification: invoice_status_update_trigger is active');
    } else {
      console.log('⚠️  Warning: Trigger not found in verification check');
    }

    // Step 2: Apply Client Email Index Optimization
    console.log('\n📈 Step 2: Applying Client Email Index Optimization...');
    const indexSQL = fs.readFileSync(
      path.join(__dirname, '02_client_email_index.sql'),
      'utf8'
    );
    
    await prisma.$executeRawUnsafe(indexSQL);
    console.log('✅ Email index optimization applied successfully!');
    
    // Verify index creation
    const indexCheck = await prisma.$queryRaw`
      SELECT indexname, indexdef 
      FROM pg_indexes 
      WHERE tablename = 'Client' AND indexname LIKE 'idx_client_email%'
      ORDER BY indexname
    `;
    
    console.log(`✅ Index verification: ${indexCheck.length} email indexes created:`);
    indexCheck.forEach(index => {
      console.log(`   - ${index.indexname}`);
    });

    // Step 3: Test the implementations
    console.log('\n🧪 Step 3: Running basic tests...');
    
    // Test trigger tables exist
    const historyTableExists = await prisma.$queryRaw`
      SELECT EXISTS (
        SELECT FROM information_schema.tables 
        WHERE table_name = 'InvoiceStatusHistory'
      )
    `;
    
    const notificationsTableExists = await prisma.$queryRaw`
      SELECT EXISTS (
        SELECT FROM information_schema.tables 
        WHERE table_name = 'InvoiceNotifications'
      )
    `;
    
    console.log(`✅ InvoiceStatusHistory table exists: ${historyTableExists[0].exists}`);
    console.log(`✅ InvoiceNotifications table exists: ${notificationsTableExists[0].exists}`);
    
    // Test email query performance (if Client table has data)
    const clientCount = await prisma.client.count();
    if (clientCount > 0) {
      console.log(`✅ Client table has ${clientCount} records - indexes ready for optimization`);
      
      // Show query plan for email lookup
      const queryPlan = await prisma.$queryRaw`
        EXPLAIN (FORMAT JSON) 
        SELECT * FROM "Client" WHERE email = 'test@example.com'
      `;
      
      const planText = JSON.stringify(queryPlan[0], null, 2);
      if (planText.includes('Index Scan')) {
        console.log('✅ Email queries are using index optimization');
      } else {
        console.log('ℹ️  Email queries will use indexes when data grows larger');
      }
    } else {
      console.log('ℹ️  Client table is empty - indexes ready for when data is added');
    }

    console.log('\n🎉 Advanced SQL Features Applied Successfully!');
    console.log('\n📋 What was implemented:');
    console.log('   • Complex invoice status trigger with business logic');
    console.log('   • Multiple email index strategies for optimal performance');
    console.log('   • Invoice status history tracking');
    console.log('   • Automated business rule notifications');
    console.log('   • Performance monitoring capabilities');
    
    console.log('\n🔍 Next Steps:');
    console.log('   • Check sql/README.md for detailed documentation');
    console.log('   • Update invoice statuses to test the trigger');
    console.log('   • Monitor query performance with EXPLAIN ANALYZE');
    console.log('   • Review InvoiceStatusHistory and InvoiceNotifications tables');

  } catch (error) {
    console.error('❌ Error applying advanced SQL features:');
    console.error(error.message);
    
    if (error.message.includes('permission')) {
      console.log('\n💡 Tip: Make sure your database user has sufficient privileges:');
      console.log('   • CREATE TABLE');
      console.log('   • CREATE INDEX');
      console.log('   • CREATE FUNCTION');
      console.log('   • CREATE TRIGGER');
    }
    
    process.exit(1);
  } finally {
    await prisma.$disconnect();
  }
}

// Run the script
if (require.main === module) {
  applyAdvancedSQL();
}

module.exports = { applyAdvancedSQL }; 