# Implementation Approaches - Advanced SQL Features

This guide explains the different ways to implement advanced SQL features in your Prisma project and when to use each approach.

## 🎯 Three Implementation Approaches

### 1. Integrated with Seed (`prisma/seed.js`) ✅ RECOMMENDED

**Best for**: Development and initial setup

```bash
npm run db:seed
# This will:
# 1. Seed your database with initial data
# 2. Apply advanced SQL features (triggers + indexes)
# 3. Verify everything is working
```

**Pros**:

- ✅ One command does everything
- ✅ Perfect for development setup
- ✅ Ensures SQL features are applied with initial data
- ✅ Non-blocking (continues if SQL features fail)

**Cons**:

- ⚠️ Re-runs SQL features every time you seed
- ⚠️ Might not be suitable for production deployments

### 2. Separate Script (`sql/apply-advanced-sql.js`)

**Best for**: Production deployments and maintenance

```bash
node sql/apply-advanced-sql.js
```

**Pros**:

- ✅ Dedicated purpose - only applies SQL features
- ✅ Can be run independently of seeding
- ✅ Better for production deployment scripts
- ✅ More granular control

**Cons**:

- ⚠️ Requires running multiple commands
- ⚠️ Need to remember to run it separately

### 3. Prisma Migration (Most Professional) 🏆 BEST FOR PRODUCTION

**Best for**: Production environments and team collaboration

```bash
# Create migration
npx prisma migrate dev --create-only --name advanced-sql-features

# Copy SQL content to migration file, then apply
npx prisma migrate dev
```

**Pros**:

- ✅ Version controlled database changes
- ✅ Automatic deployment with migrate deploy
- ✅ Rollback capabilities
- ✅ Team collaboration friendly
- ✅ Production-ready

**Cons**:

- ⚠️ More setup steps initially
- ⚠️ Need to manually copy SQL to migration files

## 📋 What Goes Where?

### ✅ Can be in `schema.prisma`:

```prisma
model Client {
  email String @unique

  // Basic indexes - Prisma handles these
  @@index([email])
  @@index([email, tenantId])
  @@index([status])
}
```

### ❌ Must be Raw SQL:

- Complex triggers with business logic
- Partial indexes (`WHERE status = 'ACTIVE'`)
- Functional indexes (`LOWER(email)`)
- Database functions
- Advanced constraints

## 🚀 Recommended Setup by Environment

### Development Environment

```bash
# Option 1: All-in-one (easiest)
npm run db:seed

# Option 2: Step by step
npx prisma migrate dev
node sql/apply-advanced-sql.js
```

### Production Environment

```bash
# Create proper migration (one-time setup)
npx prisma migrate dev --create-only --name advanced-sql-features
# Copy content from sql/*.sql files to the migration
npx prisma migrate deploy

# Or use separate deployment script
node sql/apply-advanced-sql.js
```

## 🔧 Current Implementation Status

Based on your project setup:

### ✅ What's Implemented:

1. **Basic Indexes in Prisma Schema** (`schema.prisma`)

   ```prisma
   @@index([email])                    // Basic email index
   @@index([email, tenantId])          // Composite index
   @@index([status])                   // Status index
   ```

2. **Advanced SQL Features** (`sql/` directory)

   - Complex invoice status trigger
   - Advanced email indexes (partial, functional, composite)
   - Notification system
   - Status history tracking

3. **Integrated Seeding** (`prisma/seed.js`)

   - Applies both data seeding AND SQL features
   - Non-blocking (continues if SQL fails)
   - Perfect for development

4. **Standalone Script** (`sql/apply-advanced-sql.js`)
   - For production deployments
   - Detailed verification and testing

## 📝 Usage Instructions

### For Development:

```bash
# Reset and setup everything
npx prisma migrate reset
npm run db:seed

# This gives you:
# ✅ Clean database
# ✅ Initial data
# ✅ Advanced SQL features
# ✅ Verification output
```

### For Production:

```bash
# Method 1: Separate script
npx prisma migrate deploy
node sql/apply-advanced-sql.js

# Method 2: Migration approach
npx prisma migrate deploy  # (with SQL content in migration files)
```

## 🔍 Verification

After running any approach, verify with:

```sql
-- Check trigger exists
SELECT tgname FROM pg_trigger WHERE tgname = 'invoice_status_update_trigger';

-- Check indexes exist
SELECT indexname FROM pg_indexes
WHERE tablename = 'Client' AND indexname LIKE 'idx_client_email%';

-- Test trigger
UPDATE "Invoice" SET status = 'PAID' WHERE invoice_number = 'some-invoice';
SELECT * FROM "InvoiceStatusHistory" ORDER BY changed_at DESC LIMIT 1;
```

## 🎓 Learning Outcomes

This setup teaches:

- **Prisma Limitations**: What can/cannot be done in schema.prisma
- **Raw SQL Integration**: How to extend Prisma with custom SQL
- **Deployment Strategies**: Different approaches for different environments
- **Database Optimization**: Advanced indexing and trigger strategies

## 💡 Recommendations

1. **Use integrated seeding** during development
2. **Create proper migrations** for production
3. **Keep raw SQL in separate files** for maintainability
4. **Document everything** (like this guide!)
5. **Test thoroughly** in staging before production

The current setup gives you flexibility to choose the best approach for your needs!
