-- =====================================================
-- BUSINESS INTELLIGENCE VIEWS FOR AGENCY OS
-- =====================================================
-- This file contains multiple SQL VIEWs that provide
-- business insights and simplify complex queries
-- =====================================================

-- =====================================================
-- 1. INVOICE ANALYTICS VIEW
-- =====================================================
-- Provides comprehensive invoice analytics with calculated fields
-- This view combines invoice data with client and tenant information
-- and calculates business metrics like days overdue, payment status, etc.

CREATE OR REPLACE VIEW v_invoice_analytics AS
SELECT 
    i.id,
    i.invoice_number,
    i.invoice_status,
    i.invoice_currency,
    i.invoice_total_amount,
    i.invoice_sub_total,
    i.invoice_taxes,
    i.invoice_discount,
    i.invoice_issue_date,
    i.invoice_due_date,
    i.created_at,
    i.updated_at,
    
    -- Client Information
    c.name as client_name,
    c.email as client_email,
    c.client_status,
    c.client_representative,
    
    -- Tenant Information
    t.company_name as tenant_name,
    t.company_email as tenant_email,
    
    -- Calculated Business Metrics
    CASE 
        WHEN i.invoice_status = 'OVERDUE' THEN 
            EXTRACT(DAYS FROM (NOW() - i.invoice_due_date))
        ELSE NULL 
    END as days_overdue,
    
    CASE 
        WHEN i.invoice_status = 'OVERDUE' THEN 
            EXTRACT(DAYS FROM (NOW() - i.invoice_issue_date))
        ELSE 
            EXTRACT(DAYS FROM (NOW() - i.invoice_issue_date))
    END as days_since_issue,
    
    CASE 
        WHEN i.invoice_status = 'PAID' THEN 'PAID'
        WHEN i.invoice_status = 'OVERDUE' THEN 'OVERDUE'
        WHEN i.invoice_due_date < NOW() AND i.invoice_status IN ('PENDING', 'PROCESSING') THEN 'OVERDUE'
        ELSE 'CURRENT'
    END as payment_status,
    
    -- Revenue categorization
    CASE 
        WHEN i.invoice_total_amount >= 10000 THEN 'HIGH_VALUE'
        WHEN i.invoice_total_amount >= 1000 THEN 'MEDIUM_VALUE'
        ELSE 'LOW_VALUE'
    END as value_category,
    
    -- Age categorization
    CASE 
        WHEN i.invoice_status = 'PAID' THEN 'PAID'
        WHEN EXTRACT(DAYS FROM (NOW() - i.invoice_issue_date)) <= 30 THEN 'RECENT'
        WHEN EXTRACT(DAYS FROM (NOW() - i.invoice_issue_date)) <= 90 THEN 'AGING'
        ELSE 'OLD'
    END as age_category

FROM "Invoice" i
JOIN "Client" c ON i.client_id = c.id
JOIN "Tenant" t ON i.tenant_id = t.id;

-- =====================================================
-- 2. CLIENT FINANCIAL SUMMARY VIEW
-- =====================================================
-- Provides comprehensive financial summary for each client
-- Shows total invoiced, paid, outstanding, and overdue amounts

CREATE OR REPLACE VIEW v_client_financial_summary AS
SELECT 
    c.id as client_id,
    c.name as client_name,
    c.email as client_email,
    c.client_status,
    c.client_representative,
    t.company_name as tenant_name,
    
    -- Invoice Counts
    COUNT(i.id) as total_invoices,
    COUNT(CASE WHEN i.invoice_status = 'PAID' THEN 1 END) as paid_invoices,
    COUNT(CASE WHEN i.invoice_status = 'PENDING' THEN 1 END) as pending_invoices,
    COUNT(CASE WHEN i.invoice_status = 'OVERDUE' THEN 1 END) as overdue_invoices,
    COUNT(CASE WHEN i.invoice_status = 'PROCESSING' THEN 1 END) as processing_invoices,
    COUNT(CASE WHEN i.invoice_status = 'DRAFT' THEN 1 END) as draft_invoices,
    
    -- Financial Totals
    COALESCE(SUM(i.invoice_total_amount), 0) as total_invoiced,
    COALESCE(SUM(CASE WHEN i.invoice_status = 'PAID' THEN i.invoice_total_amount ELSE 0 END), 0) as total_paid,
    COALESCE(SUM(CASE WHEN i.invoice_status IN ('PENDING', 'PROCESSING', 'OVERDUE') THEN i.invoice_total_amount ELSE 0 END), 0) as total_outstanding,
    COALESCE(SUM(CASE WHEN i.invoice_status = 'OVERDUE' THEN i.invoice_total_amount ELSE 0 END), 0) as total_overdue,
    
    -- Average values
    COALESCE(AVG(i.invoice_total_amount), 0) as average_invoice_amount,
    COALESCE(AVG(CASE WHEN i.invoice_status = 'PAID' THEN i.invoice_total_amount END), 0) as average_paid_amount,
    
    -- Date ranges
    MIN(i.invoice_issue_date) as first_invoice_date,
    MAX(i.invoice_issue_date) as last_invoice_date,
    
    -- Risk indicators
    CASE 
        WHEN COUNT(CASE WHEN i.invoice_status = 'OVERDUE' THEN 1 END) > 3 THEN 'HIGH_RISK'
        WHEN COUNT(CASE WHEN i.invoice_status = 'OVERDUE' THEN 1 END) > 1 THEN 'MEDIUM_RISK'
        ELSE 'LOW_RISK'
    END as risk_level,
    
    -- Payment behavior
    CASE 
        WHEN COUNT(i.id) = 0 THEN 'NO_HISTORY'
        WHEN COUNT(CASE WHEN i.invoice_status = 'PAID' THEN 1 END)::FLOAT / COUNT(i.id) >= 0.9 THEN 'EXCELLENT'
        WHEN COUNT(CASE WHEN i.invoice_status = 'PAID' THEN 1 END)::FLOAT / COUNT(i.id) >= 0.7 THEN 'GOOD'
        WHEN COUNT(CASE WHEN i.invoice_status = 'PAID' THEN 1 END)::FLOAT / COUNT(i.id) >= 0.5 THEN 'FAIR'
        ELSE 'POOR'
    END as payment_behavior

FROM "Client" c
LEFT JOIN "Invoice" i ON c.id = i.client_id
JOIN "Tenant" t ON c.tenant_id = t.id
GROUP BY c.id, c.name, c.email, c.client_status, c.client_representative, t.company_name;

-- =====================================================
-- CREATE INDEXES FOR VIEW PERFORMANCE
-- =====================================================

-- Indexes to optimize view performance
CREATE INDEX IF NOT EXISTS idx_invoice_analytics_status ON "Invoice"(invoice_status);
CREATE INDEX IF NOT EXISTS idx_invoice_analytics_due_date ON "Invoice"(invoice_due_date);
CREATE INDEX IF NOT EXISTS idx_invoice_analytics_issue_date ON "Invoice"(invoice_issue_date);
CREATE INDEX IF NOT EXISTS idx_invoice_analytics_client_id ON "Invoice"(client_id);
CREATE INDEX IF NOT EXISTS idx_invoice_analytics_tenant_id ON "Invoice"(tenant_id);

CREATE INDEX IF NOT EXISTS idx_client_financial_status ON "Client"(client_status);
CREATE INDEX IF NOT EXISTS idx_client_financial_tenant ON "Client"(tenant_id);
