-- This trigger fires when an invoice status is updated

-- First, create a table to log invoice status changes and history
CREATE TABLE IF NOT EXISTS "InvoiceStatusHistory" (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    invoice_id VARCHAR(255) NOT NULL,
    old_status VARCHAR(50),
    new_status VARCHAR(50) NOT NULL,
    changed_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    total_amount FLOAT,
    days_since_issue INTEGER,
    auto_generated BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (invoice_id) REFERENCES "Invoice"(id) ON DELETE CASCADE
);

-- Create a notifications table
CREATE TABLE IF NOT EXISTS "InvoiceNotifications" (
    id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    invoice_id VARCHAR(255) NOT NULL,
    notification_type VARCHAR(50) NOT NULL,
    message TEXT NOT NULL,
    priority VARCHAR(20) DEFAULT 'MEDIUM',
    created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
    processed BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (invoice_id) REFERENCES "Invoice"(id) ON DELETE CASCADE
);

-- Trigger function that performs multiple operations
CREATE OR REPLACE FUNCTION handle_invoice_status_update()
RETURNS TRIGGER AS $$
DECLARE
    days_since_issue INTEGER;
    client_total_outstanding FLOAT := 0;
    client_invoice_count INTEGER := 0;
    tenant_monthly_revenue FLOAT := 0;
    notification_message TEXT;
    priority_level VARCHAR(20) := 'MEDIUM';
BEGIN
    -- Only proceed if status has actually changed
    IF OLD.invoice_status = NEW.invoice_status THEN
        RETURN NEW;
    END IF;

    -- Calculate days since invoice was issued
    days_since_issue := EXTRACT(DAYS FROM (NOW() - NEW.invoice_issue_date));

    -- Log the status change in history
    INSERT INTO "InvoiceStatusHistory" (
        invoice_id, 
        old_status, 
        new_status, 
        total_amount, 
        days_since_issue,
        auto_generated
    ) VALUES (
        NEW.id, 
        OLD.invoice_status, 
        NEW.invoice_status, 
        NEW.invoice_total_amount, 
        days_since_issue,
        TRUE
    );

    -- Calculate client's total outstanding invoices
    SELECT 
        COALESCE(SUM(invoice_total_amount), 0),
        COUNT(*)
    INTO client_total_outstanding, client_invoice_count
    FROM "Invoice" 
    WHERE "client_id" = NEW."client_id" 
    AND invoice_status IN ('PENDING', 'OVERDUE', 'PROCESSING');

    -- Calculate tenant's monthly revenue for current month
    SELECT COALESCE(SUM(invoice_total_amount), 0)
    INTO tenant_monthly_revenue
    FROM "Invoice"
    WHERE "tenant_id" = NEW."tenant_id"
    AND invoice_status = 'PAID'
    AND EXTRACT(MONTH FROM invoice_issue_date) = EXTRACT(MONTH FROM NOW())
    AND EXTRACT(YEAR FROM invoice_issue_date) = EXTRACT(YEAR FROM NOW());

    -- Complex business logic based on status changes
    CASE NEW.invoice_status
        WHEN 'PAID' THEN
            notification_message := format(
                'Invoice %s (€%s) has been PAID! Client total outstanding: €%s across %s invoices. Monthly revenue: €%s',
                NEW.invoice_number, 
                to_char(NEW.invoice_total_amount, 'FM999999999.00'),
                to_char(client_total_outstanding, 'FM999999999.00'),
                client_invoice_count,
                to_char(tenant_monthly_revenue, 'FM999999999.00')
            );
            priority_level := 'LOW';
            
            -- Update client status if they have no more outstanding invoices
            IF client_total_outstanding = 0 THEN
                UPDATE "Client" 
                SET client_status = 'ACTIVE'
                WHERE id = NEW."client_id" AND client_status != 'ACTIVE';
                
                notification_message := notification_message || ' Client status updated to ACTIVE.';
            END IF;

        WHEN 'OVERDUE' THEN
            notification_message := format(
                'CRITICAL: Invoice %s (€%s) is now OVERDUE (%s days since issue). Client owes €%s total across %s invoices.',
                NEW.invoice_number,
                to_char(NEW.invoice_total_amount, 'FM999999999.00'),
                days_since_issue,
                to_char(client_total_outstanding, 'FM999999999.00'),
                client_invoice_count
            );
            priority_level := 'HIGH';
            
            -- Flag client if they have too many overdue invoices
            IF client_invoice_count > 3 THEN
                UPDATE "Client" 
                SET client_status = 'INACTIVE'
                WHERE id = NEW."client_id";
                
                notification_message := notification_message || ' Client flagged as INACTIVE due to multiple overdue invoices.';
            END IF;

        WHEN 'PROCESSING' THEN
            notification_message := format(
                'Invoice %s (€%s) is now being processed. Days since issue: %s',
                NEW.invoice_number,
                to_char(NEW.invoice_total_amount, 'FM999999999.00'),
                days_since_issue
            );

        WHEN 'PENDING' THEN
            notification_message := format(
                'Invoice %s (€%s) is pending payment. Client total outstanding: €%s',
                NEW.invoice_number,
                to_char(NEW.invoice_total_amount, 'FM999999999.00'),
                to_char(client_total_outstanding, 'FM999999999.00')
            );
            
            -- Warn if client has high outstanding amount
            IF client_total_outstanding > 10000 THEN
                priority_level := 'HIGH';
                notification_message := notification_message || ' WARNING: High outstanding amount!';
            END IF;

        ELSE
            notification_message := format(
                'Invoice %s status changed from %s to %s',
                NEW.invoice_number, OLD.invoice_status, NEW.invoice_status
            );
    END CASE;

    -- Insert notification
    INSERT INTO "InvoiceNotifications" (
        invoice_id,
        notification_type,
        message,
        priority
    ) VALUES (
        NEW.id,
        'STATUS_CHANGE',
        notification_message,
        priority_level
    );

    -- Auto-update due date if status changes to PENDING and due date is in the past
    IF NEW.invoice_status = 'PENDING' AND NEW.invoice_due_date < NOW() THEN
        NEW.invoice_due_date := NOW() + INTERVAL '30 days';
        
        INSERT INTO "InvoiceNotifications" (
            invoice_id,
            notification_type,
            message,
            priority
        ) VALUES (
            NEW.id,
            'DUE_DATE_EXTENDED',
            format('Due date automatically extended to %s for invoice %s', NEW.invoice_due_date::DATE, NEW.invoice_number),
            'MEDIUM'
        );
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Create the trigger
DROP TRIGGER IF EXISTS invoice_status_update_trigger ON "Invoice";
CREATE TRIGGER invoice_status_update_trigger
    AFTER UPDATE ON "Invoice"
    FOR EACH ROW
    WHEN (OLD.invoice_status IS DISTINCT FROM NEW.invoice_status)
    EXECUTE FUNCTION handle_invoice_status_update();

-- Create indexes for performance on new tables
CREATE INDEX IF NOT EXISTS idx_invoice_status_history_invoice_id ON "InvoiceStatusHistory"(invoice_id);
CREATE INDEX IF NOT EXISTS idx_invoice_status_history_changed_at ON "InvoiceStatusHistory"(changed_at);
CREATE INDEX IF NOT EXISTS idx_invoice_notifications_invoice_id ON "InvoiceNotifications"(invoice_id);
CREATE INDEX IF NOT EXISTS idx_invoice_notifications_processed ON "InvoiceNotifications"(processed) WHERE processed = FALSE; 