import { Flex, Text, Button, useToast, useDisclosure } from "@chakra-ui/react";
import { Link, useNavigate } from "react-router-dom";
import {
    AlertDialog,
    AlertDialogBody,
    AlertDialogFooter,
    AlertDialogHeader,
    AlertDialogContent,
    AlertDialogOverlay,
} from "@chakra-ui/react";
import { useRef, useState } from "react";

const SoldBlogCard = ({
    vin,
    vehicle_body,
    car_make,
    car_model,
    p_year,
    capacity,
    power,
    color,
    price,
    status,
    onVehicleDeleted,
}) => {
    const navigate = useNavigate();
    const toast = useToast();
    const { isOpen, onOpen, onClose } = useDisclosure();
    const cancelRef = useRef();
    const [isDeleting, setIsDeleting] = useState(false);

    const handleSetAsSold = () => {
        const vehicleInfo = {
            vin,
            make: car_make,
            model: car_model,
            p_year: p_year.slice(1, -1),
            price: price.slice(1, -1)
        };
        navigate("/createagreement", { state: { vin, vehicleInfo } });
    };

    const handleViewDocuments = () => {
        navigate("/documents", { state: { vin } });
    };

    const handleDeleteVehicle = async () => {
        setIsDeleting(true);
        try {
            const response = await fetch(`http://localhost:3001/vehicle/${vin}`, {
                method: "DELETE",
                credentials: 'include',
            });

            if (response.ok) {
                toast({
                    title: "Vehicle Deleted",
                    description: "Vehicle has been successfully deleted from your inventory.",
                    status: "success",
                    duration: 3000,
                    isClosable: true,
                });
                onClose();
                // Call the callback to refresh the vehicle list
                if (onVehicleDeleted) {
                    onVehicleDeleted();
                }
            } else {
                const errorData = await response.json();
                toast({
                    title: "Error",
                    description: errorData.error || "Failed to delete vehicle",
                    status: "error",
                    duration: 3000,
                    isClosable: true,
                });
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "An error occurred while deleting the vehicle",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setIsDeleting(false);
        }
    };
    return (
        <Flex
            minWidth='400px'
            boxShadow='lg'
            hover={{ boxShadow: "xl" }}
            borderBottomRadius='10px'
            bg='white'
            borderRadius='13px'
            transition='all 0.4s ease'
            padding='20px'
            maxWidth='400px'
            direction='column'
            borderTop='4px'
            borderColor='teal'
            _hover={{ boxShadow: "2xl" }}>
            <Text
                marginBottom='10px'
                fontWeight='bold'
                fontSize='20px'
                align='center'>
                {car_make + " " + car_model}
            </Text>
            <Flex width='100%' height='2px' bg='teal' />
            <Flex
                width='100%'
                justifyItems='center'
                alignItems='center'
                justifyContent='space-around'>
                <Flex
                    direction='column'
                    width='100%'
                    maxWidth='120px'
                    alignItems='center'>
                    <Text fontSize='14px' padding='5px'>
                        {vehicle_body}
                    </Text>
                    <Flex height='2px' bg='teal' width='100%'></Flex>
                    <Text fontSize='14px' padding='5px'>
                        {p_year.slice(1, -1)}
                    </Text>
                </Flex>
                <Flex
                    maxWidth='120px'
                    marginTop='5px'
                    height='90%'
                    width='2px'
                    bg='teal'></Flex>
                <Flex direction='column' alignItems='center'>
                    <Text fontSize='14px' padding='5px'>
                        {capacity}
                    </Text>
                    <Flex height='2px' bg='teal' width='100%'></Flex>
                    <Text fontSize='14px' padding='5px'>
                        {color}
                    </Text>
                </Flex>
            </Flex>
            <Flex marginTop='5px' width='100%' height='2px' bg='teal' />
            <Flex width='100%' justifyContent='center' direction='column'>
                <Text
                    marginTop='10px'
                    fontSize='20px'
                    fontWeight='bold'
                    textAlign='center'
                    justifySelf='right'>
                    {price.slice(1, -1) + "e"}
                </Text>
                <Flex marginTop='5px' justifyContent='space-between'>
                    {status == "false" ? (
                        <Button
                            minW="120px"
                            bg='#004d42'
                            color='white'
                            _hover={{
                                background: "#006b5c",
                            }}>
                            Sold
                        </Button>
                    ) : (
                        <Button
                            bg='#ff4a6e'
                            color='white'
                            _hover={{
                                background: "#fa5c7c",
                            }}
                            onClick={handleSetAsSold}>
                            Set as Sold
                        </Button>
                    )}
                    {status == "false" ? (
                        <Button
                            bg='blue'
                            color='white'
                            _hover={{
                                background: "#1976d2",
                            }}
                            onClick={handleViewDocuments}>
                            Documents
                        </Button>
                    ) : (
                        <Button
                            bg='red'
                            color='white'
                            _hover={{
                                background: "#d32f2f",
                            }}
                            onClick={onOpen}>
                            Delete
                        </Button>
                    )}
                </Flex>
            </Flex>
            
            {/* Delete Confirmation Dialog */}
            <AlertDialog
                isOpen={isOpen}
                leastDestructiveRef={cancelRef}
                onClose={onClose}
            >
                <AlertDialogOverlay>
                    <AlertDialogContent>
                        <AlertDialogHeader fontSize="lg" fontWeight="bold">
                            Delete Vehicle
                        </AlertDialogHeader>

                        <AlertDialogBody>
                            Are you sure you want to delete this vehicle? This action cannot be undone.
                            <br />
                            <br />
                            <strong>{car_make} {car_model} ({p_year.slice(1, -1)})</strong>
                        </AlertDialogBody>

                        <AlertDialogFooter>
                            <Button ref={cancelRef} onClick={onClose}>
                                Cancel
                            </Button>
                            <Button 
                                colorScheme="red" 
                                onClick={handleDeleteVehicle} 
                                ml={3}
                                isLoading={isDeleting}
                                loadingText="Deleting...">
                                Delete
                            </Button>
                        </AlertDialogFooter>
                    </AlertDialogContent>
                </AlertDialogOverlay>
            </AlertDialog>
        </Flex>
    );
};

export default SoldBlogCard;
