import { Formik, Field, Form } from "formik";
import * as Yup from "yup";
import { Button, Text, Flex, useToast } from "@chakra-ui/react";
import { useNavigate, useLocation } from "react-router-dom";

const SignupSchema = Yup.object().shape({
    A_Id: Yup.string()
        .required("This field is required!")
        .min(5, "Agreement ID must be at least 5 characters"),
    Price: Yup.number()
        .required("This field is required!")
        .min(1, "Price must be greater than 0"),
    EMBG: Yup.string()
        .min(13, "Must be 13 digits exactly!")
        .max(13, "Must be 13 digits exactly!")
        .required("This field is required!")
        .matches(/^[0-9]*$/, "Must be a number."),
    Datum: Yup.date()
        .required("This field is required!"),
});

const CreateAgreement = () => {
    const navigate = useNavigate();
    const location = useLocation();
    const toast = useToast();
    
    // Get VIN from location state
    const { vin, vehicleInfo } = location.state || {};

    const createAgreement = async ({ A_Id, Price, EMBG, Datum }) => {
        try {
            const response = await fetch("http://localhost:3001/agreement", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                credentials: 'include',
                body: JSON.stringify({
                    A_Id,
                    Price,
                    Status: true,
                    Datum,
                    VIN: vin,
                    EMBG
                }),
            });

            const data = await response.text();
            
            if (response.ok) {
                toast({
                    title: "Agreement created",
                    description: "Agreement has been created successfully!",
                    status: "success",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/mylist");
            } else {
                toast({
                    title: "Error",
                    description: "Failed to create agreement",
                    status: "error",
                    duration: 3000,
                    isClosable: true,
                });
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "An error occurred while creating agreement",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        }
    };

    if (!vin) {
        return (
            <Flex direction='column' align='center'>
                <Text fontSize='24px' color='red'>No vehicle selected</Text>
                <Button onClick={() => navigate("/mylist")} marginTop='20px'>
                    Back to My Vehicles
                </Button>
            </Flex>
        );
    }

    return (
        <>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='32px'
                justifySelf='center'>
                Create Agreement
            </Text>
            
            {vehicleInfo && (
                <Flex
                    bg='gray.100'
                    padding='15px'
                    borderRadius='8px'
                    marginBottom='20px'
                    direction='column'>
                    <Text fontWeight='bold'>Vehicle Information:</Text>
                    <Text>VIN: {vehicleInfo.vin}</Text>
                    <Text>Make: {vehicleInfo.make}</Text>
                    <Text>Model: {vehicleInfo.model}</Text>
                    <Text>Year: {vehicleInfo.p_year}</Text>
                    <Text>Price: ${vehicleInfo.price}</Text>
                </Flex>
            )}

            <Formik
                initialValues={{
                    A_Id: "",
                    Price: vehicleInfo?.price || "",
                    EMBG: "",
                    Datum: new Date().toISOString().split('T')[0],
                }}
                validationSchema={SignupSchema}
                onSubmit={async (values) => {
                    await createAgreement(values);
                }}>
                {({ isSubmitting, errors, touched }) => (
                    <Form width='100%'>
                        <Flex
                            boxShadow='lg'
                            hover={{ boxShadow: "xl" }}
                            borderBottomRadius='10px'
                            bg='white'
                            borderRadius='13px'
                            transition='all 0.4s ease'
                            direction='column'
                            padding='20px'
                            width='100%'>
                            
                            <Flex padding='10px' direction='column'>
                                <Flex>
                                    <label
                                        style={{ marginRight: "15px", minWidth: "120px" }}
                                        htmlFor='A_Id'>
                                        Agreement ID
                                    </label>
                                    <Field
                                        name='A_Id'
                                        placeholder='AG-2024-001'
                                    />
                                </Flex>
                                {errors.A_Id && touched.A_Id ? (
                                    <Flex color='red'>{errors.A_Id}</Flex>
                                ) : null}
                            </Flex>

                            <Flex padding='10px' direction='column'>
                                <Flex>
                                    <label
                                        style={{ marginRight: "15px", minWidth: "120px" }}
                                        htmlFor='Price'>
                                        Agreement Price
                                    </label>
                                    <Field
                                        name='Price'
                                        placeholder='35000'
                                        type='number'
                                    />
                                </Flex>
                                {errors.Price && touched.Price ? (
                                    <Flex color='red'>{errors.Price}</Flex>
                                ) : null}
                            </Flex>

                            <Flex padding='10px' direction='column'>
                                <Flex>
                                    <label
                                        style={{ marginRight: "15px", minWidth: "120px" }}
                                        htmlFor='EMBG'>
                                        Client EMBG
                                    </label>
                                    <Field
                                        name='EMBG'
                                        placeholder='1507995410010'
                                    />
                                </Flex>
                                {errors.EMBG && touched.EMBG ? (
                                    <Flex color='red'>{errors.EMBG}</Flex>
                                ) : null}
                            </Flex>

                            <Flex padding='10px' direction='column'>
                                <Flex>
                                    <label
                                        style={{ marginRight: "15px", minWidth: "120px" }}
                                        htmlFor='Datum'>
                                        Agreement Date
                                    </label>
                                    <Field
                                        name='Datum'
                                        type='date'
                                    />
                                </Flex>
                                {errors.Datum && touched.Datum ? (
                                    <Flex color='red'>{errors.Datum}</Flex>
                                ) : null}
                            </Flex>

                            <Flex direction='row' gap='10px' marginTop='20px'>
                                <Button
                                    colorScheme='blue'
                                    type='submit'
                                    disabled={isSubmitting}>
                                    Create Agreement
                                </Button>
                                <Button
                                    colorScheme='gray'
                                    onClick={() => navigate("/mylist")}>
                                    Cancel
                                </Button>
                            </Flex>
                        </Flex>
                    </Form>
                )}
            </Formik>
        </>
    );
};

export default CreateAgreement;
