import { Formik, Form, Field } from "formik";
import { Button, Flex, Text, useToast, Box } from "@chakra-ui/react";
import { Input, InputGroup, InputLeftElement } from "@chakra-ui/react";
import { InfoIcon } from "@chakra-ui/icons";
import { useNavigate, useLocation } from "react-router-dom";
import * as Yup from "yup";

const SignupSchema = Yup.object().shape({
    Bank: Yup.string()
        .required("Bank name is required")
        .min(2, "Bank name must be at least 2 characters"),
    IBAN: Yup.string()
        .required("IBAN is required")
        .min(15, "IBAN must be at least 15 characters")
        .matches(/^[A-Z]{2}[0-9]{2}[A-Z0-9]+$/, "Invalid IBAN format"),
});

const CreatePayment = () => {
    const navigate = useNavigate();
    const location = useLocation();
    const toast = useToast();
    
    // Get agreement from location state
    const { agreement } = location.state || {};

    const createPayment = async ({ Bank, IBAN }) => {
        try {
            const response = await fetch("http://localhost:3001/payment", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                credentials: 'include',
                body: JSON.stringify({
                    Bank,
                    IBAN,
                    Amount: agreement.price, // Fixed amount from agreement
                    A_Id: agreement.a_id
                }),
            });

            const data = await response.text();
            
            if (response.ok) {
                toast({
                    title: "Payment created",
                    description: "Payment has been processed successfully!",
                    status: "success",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/myagreements");
            } else {
                toast({
                    title: "Error",
                    description: "Failed to create payment",
                    status: "error",
                    duration: 3000,
                    isClosable: true,
                });
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "An error occurred while processing payment",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        }
    };

    if (!agreement) {
        return (
            <Flex direction='column' align='center'>
                <Text fontSize='24px' color='red'>No agreement selected</Text>
                <Button onClick={() => navigate("/myagreements")} marginTop='20px'>
                    Back to My Agreements
                </Button>
            </Flex>
        );
    }

    return (
        <Flex width='100%' flexDir='column' justifyContent='center' maxWidth='500px'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                Make Payment
            </Text>
            
            {/* Agreement Information */}
            <Box
                borderWidth="1px"
                borderRadius="lg"
                overflow="hidden"
                p="6"
                bg="blue.50"
                boxShadow="md"
                marginBottom="6">
                
                <Flex direction="column" gap="2">
                    <Text fontWeight="bold" fontSize="lg" mb="2">
                        Agreement Information
                    </Text>
                    <Text>Agreement ID: {agreement.a_id}</Text>
                    <Text>Vehicle: {agreement.make} {agreement.model} ({agreement.p_year})</Text>
                    <Text>Dealership: {agreement.dealership_name}</Text>
                    <Text fontWeight="bold" color="green.600" fontSize="lg">
                        Amount Due: ${agreement.price}
                    </Text>
                </Flex>
            </Box>

            <Formik
                initialValues={{
                    Bank: "",
                    IBAN: "",
                }}
                validationSchema={SignupSchema}
                onSubmit={async (values, { setSubmitting }) => {
                    await createPayment(values);
                    setSubmitting(false);
                }}>
                {({ isSubmitting, errors, touched }) => (
                    <Form>
                        <Flex flexDir='column' gap='4'>
                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Bank Name *
                                </Text>
                                <Field name="Bank">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="First National Bank"
                                                isInvalid={errors.Bank && touched.Bank}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.Bank && touched.Bank && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.Bank}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    IBAN *
                                </Text>
                                <Field name="IBAN">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="MK07250120000058984"
                                                isInvalid={errors.IBAN && touched.IBAN}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.IBAN && touched.IBAN && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.IBAN}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Payment Amount
                                </Text>
                                <InputGroup>
                                    <InputLeftElement pointerEvents='none'>
                                        <InfoIcon color='gray.300' />
                                    </InputLeftElement>
                                    <Input
                                        value={`$${agreement.price}`}
                                        isReadOnly
                                        bg="gray.100"
                                        color="green.600"
                                        fontWeight="bold"
                                    />
                                </InputGroup>
                                <Text color="gray.500" fontSize="sm" marginTop="1">
                                    This amount is fixed and cannot be changed
                                </Text>
                            </Flex>

                            <Flex direction='row' gap='3' marginTop='4'>
                                <Button
                                    type="submit"
                                    isLoading={isSubmitting}
                                    loadingText="Processing..."
                                    colorScheme="green"
                                    size="lg"
                                    flex="1">
                                    Process Payment
                                </Button>
                                <Button
                                    colorScheme="gray"
                                    size="lg"
                                    onClick={() => navigate("/myagreements")}>
                                    Cancel
                                </Button>
                            </Flex>
                        </Flex>
                    </Form>
                )}
            </Formik>
        </Flex>
    );
};

export default CreatePayment;
