import { useState, useEffect } from "react";
import { Button, Flex, Box, Text, Badge, useToast, SimpleGrid } from "@chakra-ui/react";
import { useNavigate, useLocation } from "react-router-dom";

const Documents = () => {
    const [agreement, setAgreement] = useState(null);
    const [payment, setPayment] = useState(null);
    const [loading, setLoading] = useState(true);
    const navigate = useNavigate();
    const location = useLocation();
    const toast = useToast();
    
    // Get data from location state - either VIN or full agreement/payment data
    const { vin, agreementData, paymentData } = location.state || {};

    useEffect(() => {
        if (vin || agreementData || paymentData) {
            if (agreementData || paymentData) {
                // We already have the data, just set it
                if (agreementData) {
                    setAgreement(agreementData);
                    // Check if the agreement data includes payment information
                    if (agreementData.payment_exists && agreementData.p_id) {
                        // Payment data is included in the agreement
                        setPayment({
                            p_id: agreementData.p_id,
                            amount: agreementData.payment_amount,
                            bank: agreementData.bank,
                            iban: agreementData.iban,
                            a_id: agreementData.a_id
                        });
                    } else if (agreementData.a_id) {
                        // Check if there's a separate payment for it
                        fetchPaymentForAgreement(agreementData.a_id);
                    }
                }
                if (paymentData) {
                    setPayment(paymentData);
                    // If we have payment data, fetch the corresponding agreement
                    if (paymentData.a_id) {
                        fetchAgreementForPayment(paymentData.a_id);
                    }
                }
                setLoading(false);
            } else if (vin) {
                // We only have VIN, fetch the data
                fetchDocuments();
            }
        } else {
            toast({
                title: "Error",
                description: "No vehicle data provided",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
            navigate("/reports");
        }
    }, [vin, agreementData, paymentData]);

    const fetchDocuments = async () => {
        try {
            // Fetch agreement details
            const agreementResponse = await fetch(`http://localhost:3001/agreement/${vin}`, {
                credentials: 'include'
            });

            if (agreementResponse.ok) {
                const agreementData = await agreementResponse.json();
                // The backend returns an array, so take the first element
                if (Array.isArray(agreementData) && agreementData.length > 0) {
                    const agreement = agreementData[0];
                    setAgreement(agreement);
                    // Check if there's a payment for this agreement
                    if (agreement.a_id) {
                        fetchPaymentForAgreement(agreement.a_id);
                    }
                }
            }

            // Fetch payment details
            const paymentResponse = await fetch(`http://localhost:3001/payment/vin/${vin}`, {
                credentials: 'include'
            });

            if (paymentResponse.ok) {
                const paymentData = await paymentResponse.json();
                setPayment(paymentData);
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "Failed to load documents",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setLoading(false);
        }
    };

    const fetchPaymentForAgreement = async (agreementId) => {
        try {
            const response = await fetch(`http://localhost:3001/payment/${agreementId}`, {
                credentials: 'include'
            });

            if (response.ok) {
                const paymentData = await response.json();
                setPayment(paymentData);
            }
        } catch (error) {
            // Payment not found is not an error, just means no payment exists yet
            console.log('No payment found for agreement:', agreementId);
        }
    };

    const fetchAgreementForPayment = async (agreementId) => {
        try {
            const response = await fetch(`http://localhost:3001/agreement/id/${agreementId}`, {
                credentials: 'include'
            });

            if (response.ok) {
                const agreementData = await response.json();
                setAgreement(agreementData);
            }
        } catch (error) {
            console.error('Error fetching agreement for payment:', error);
        }
    };

    if (loading) {
        return (
            <Flex justify="center" align="center" height="200px">
                <Text>Loading documents...</Text>
            </Flex>
        );
    }

    return (
        <Flex width='100%' flexDir='column' justifyContent='center' maxWidth='800px'>
            <Flex justify="space-between" align="center" mb="6">
                <Text
                    marginBottom='20px'
                    textAlign='center'
                    fontWeight='bold'
                    fontSize='30px'
                    justifySelf='center'>
                    Vehicle Documents
                </Text>
                <Button
                    colorScheme="blue"
                    onClick={() => navigate("/reports")}>
                    Back to Reports
                </Button>
            </Flex>

            <Box
                borderWidth="1px"
                borderRadius="lg"
                overflow="hidden"
                p="6"
                bg="white"
                boxShadow="lg">
                
                <Flex direction="column" gap="6">
                    {/* Vehicle Information */}
                    {agreement && (
                        <Box bg="gray.50" p="4" borderRadius="md">
                            <Text fontWeight="bold" fontSize="lg" mb="3">Vehicle Information</Text>
                            <Text>VIN: {agreement.vin}</Text>
                            <Text>Make: {agreement.make}</Text>
                            <Text>Model: {agreement.model}</Text>
                            <Text>Year: {agreement.p_year}</Text>
                            <Text>Color: {agreement.color}</Text>
                        </Box>
                    )}
                    
                    {/* Agreement Information */}
                    {agreement ? (
                        <Box bg="blue.50" p="4" borderRadius="md">
                            <Flex justify="space-between" align="center" mb="3">
                                <Text fontWeight="bold" fontSize="lg">Agreement Details</Text>
                                <Badge colorScheme="blue" fontSize="md" p="2">
                                    AGREEMENT
                                </Badge>
                            </Flex>
                            <Text>Agreement ID: {agreement.a_id}</Text>
                            <Text>Date: {new Date(agreement.datum).toLocaleDateString()}</Text>
                            <Text>Price: ${agreement.price}</Text>
                        </Box>
                    ) : (
                        <Box bg="red.50" p="4" borderRadius="md">
                            <Text color="red.600" fontWeight="bold">
                                Agreement not found
                            </Text>
                        </Box>
                    )}
                    
                    {/* Payment Information */}
                    {payment ? (
                        <Box bg="green.50" p="4" borderRadius="md">
                            <Flex justify="space-between" align="center" mb="3">
                                <Text fontWeight="bold" fontSize="lg">Payment Details</Text>
                                <Badge colorScheme="green" fontSize="md" p="2">
                                    PAID
                                </Badge>
                            </Flex>
                            <Text>Payment ID: {payment.p_id}</Text>
                            <Text>Amount: ${payment.amount}</Text>
                            <Text>Bank: {payment.bank}</Text>
                            <Text>IBAN: {payment.iban}</Text>
                        </Box>
                    ) : (
                        <Box bg="yellow.50" p="4" borderRadius="md">
                            <Flex justify="space-between" align="center" mb="3">
                                <Text fontWeight="bold" fontSize="lg">Payment Status</Text>
                                <Badge colorScheme="orange" fontSize="md" p="2">
                                    PENDING
                                </Badge>
                            </Flex>
                            <Text color="orange.600" fontWeight="bold">
                                Payment not made yet
                            </Text>
                            <Text color="gray.600" mt="2">
                                This vehicle has been sold but payment has not been received yet.
                            </Text>
                        </Box>
                    )}
                    
                    {/* Parties Information */}
                    {agreement && (
                        <Box bg="purple.50" p="4" borderRadius="md">
                            <Text fontWeight="bold" fontSize="lg" mb="3">Parties</Text>
                            <Text>Dealership: {agreement.d_name}</Text>
                            <Text>Client: {agreement.c_name}</Text>
                        </Box>
                    )}
                </Flex>
            </Box>
            
            {/* Action Buttons */}
            <Flex direction="row" gap="3" marginTop="6" justify="center">
                {payment && (
                    <Button
                        colorScheme="gray"
                        onClick={() => window.print()}>
                        Print Documents
                    </Button>
                )}
            </Flex>
        </Flex>
    );
};

export default Documents;
