import { useState } from "react";
import { Formik, Form, Field } from "formik";
import { Button, Flex, Text, useToast, VStack, HStack, Box, Input, InputGroup, InputLeftElement } from "@chakra-ui/react";
import { EmailIcon, LockIcon } from "@chakra-ui/icons";
import { useNavigate } from "react-router-dom";
import * as Yup from "yup";

const Login = () => {
    const navigate = useNavigate();
    const toast = useToast();
    const [userType, setUserType] = useState(null); // 'client' or 'dealership'

    const validationSchema = Yup.object({
        email: Yup.string()
            .email("Invalid email format")
            .required("Email is required"),
        pass: Yup.string()
            .required("Password is required")
            .min(6, "Password must be at least 6 characters"),
    });

    const initialValues = {
        email: "",
        pass: "",
    };

    const handleSubmit = async (values, { setSubmitting }) => {
        if (!userType) {
            toast({
                title: "Error",
                description: "Please select whether you are a client or dealership",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
            setSubmitting(false);
            return;
        }

        try {
            const response = await fetch(`http://localhost:3001/login/${userType}`, {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                credentials: 'include',
                body: JSON.stringify(values),
            });

            if (response.ok) {
                toast({
                    title: "Login Successful",
                    description: `Welcome back!`,
                    status: "success",
                    duration: 2000,
                    isClosable: true,
                });
                
                // Navigate based on user type
                setTimeout(() => {
                    if (userType === 'dealership') {
                        navigate("/mylist");
                    } else {
                        navigate("/myagreements");
                    }
                }, 100);
            } else {
                const errorData = await response.json();
                throw new Error(errorData.error || "Login failed");
            }
        } catch (error) {
            toast({
                title: "Login Failed",
                description: error.message,
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setSubmitting(false);
        }
    };

    return (
        <Flex width='100%' flexDir='column' justifyContent='center' maxWidth='500px'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                Login
            </Text>
            
            <Box
                borderWidth="1px"
                borderRadius="lg"
                overflow="hidden"
                p="8"
                bg="white"
                boxShadow="lg">
                
                <VStack spacing={6} align="stretch">
                    {/* User Type Selection */}
                    <VStack spacing={3} align="stretch">
                        <Text fontSize="lg" fontWeight="bold" textAlign="center">
                            I am a:
                        </Text>
                        <HStack spacing={4} justify="center">
                            <Button
                                colorScheme={userType === 'client' ? 'blue' : 'gray'}
                                variant={userType === 'client' ? 'solid' : 'outline'}
                                onClick={() => setUserType('client')}
                                minW="120px">
                                Client
                            </Button>
                            <Button
                                colorScheme={userType === 'dealership' ? 'green' : 'gray'}
                                variant={userType === 'dealership' ? 'solid' : 'outline'}
                                onClick={() => setUserType('dealership')}
                                minW="120px">
                                Dealership
                            </Button>
                        </HStack>
                    </VStack>

                    {/* Login Form */}
                    <Formik
                        initialValues={initialValues}
                        validationSchema={validationSchema}
                        onSubmit={handleSubmit}>
                        {({ isSubmitting, errors, touched }) => (
                            <Form>
                                <VStack spacing={4} align="stretch">
                                    <Flex flexDir='column'>
                                        <Text marginBottom='5px' fontWeight='bold'>
                                            Email *
                                        </Text>
                                        <Field name="email">
                                            {({ field }) => (
                                                <InputGroup>
                                                    <InputLeftElement pointerEvents='none'>
                                                        <EmailIcon color='gray.300' />
                                                    </InputLeftElement>
                                                    <Input
                                                        {...field}
                                                        type="email"
                                                        placeholder="Enter your email"
                                                        isInvalid={errors.email && touched.email}
                                                    />
                                                </InputGroup>
                                            )}
                                        </Field>
                                        {errors.email && touched.email && (
                                            <Text color="red.500" fontSize="sm" marginTop="1">
                                                {errors.email}
                                            </Text>
                                        )}
                                    </Flex>

                                    <Flex flexDir='column'>
                                        <Text marginBottom='5px' fontWeight='bold'>
                                            Password *
                                        </Text>
                                        <Field name="pass">
                                            {({ field }) => (
                                                <InputGroup>
                                                    <InputLeftElement pointerEvents='none'>
                                                        <LockIcon color='gray.300' />
                                                    </InputLeftElement>
                                                    <Input
                                                        {...field}
                                                        type="password"
                                                        placeholder="Enter your password"
                                                        isInvalid={errors.pass && touched.pass}
                                                    />
                                                </InputGroup>
                                            )}
                                        </Field>
                                        {errors.pass && touched.pass && (
                                            <Text color="red.500" fontSize="sm" marginTop="1">
                                                {errors.pass}
                                            </Text>
                                        )}
                                    </Flex>

                                    <Button
                                        type="submit"
                                        isLoading={isSubmitting}
                                        loadingText="Logging in..."
                                        colorScheme={userType === 'dealership' ? 'green' : 'blue'}
                                        size="lg"
                                        marginTop="4"
                                        isDisabled={!userType}>
                                        Login
                                    </Button>
                                </VStack>
                            </Form>
                        )}
                    </Formik>
                    
                    <Text fontSize="sm" color="gray.500" textAlign="center" mt="4">
                        Don't have an account? <Button variant="link" colorScheme="blue" onClick={() => navigate("/register")}>Register here</Button>
                    </Text>
                </VStack>
            </Box>
        </Flex>
    );
};

export default Login;
