import { useState, useEffect } from "react";
import { Button, Flex, SimpleGrid, Text, Box, Badge, useToast } from "@chakra-ui/react";
import { useNavigate } from "react-router-dom";

const MyAgreements = () => {
    const [agreements, setAgreements] = useState([]);
    const [loading, setLoading] = useState(true);
    const navigate = useNavigate();
    const toast = useToast();

    // Get client agreements
    const getMyAgreements = async () => {
        try {
            const response = await fetch("http://localhost:3001/client/agreements", {
                method: "GET",
                credentials: 'include',
            });

            if (response.ok) {
                const data = await response.json();
                setAgreements(data);
            } else if (response.status === 401) {
                toast({
                    title: "Unauthorized",
                    description: "Please log in as a client to view agreements",
                    status: "error",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/clientlogin");
            } else {
                throw new Error("Failed to fetch agreements");
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "Failed to load agreements",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        getMyAgreements();
    }, []);

    const handleMakePayment = (agreement) => {
        navigate("/createpayment", { state: { agreement } });
    };

    const handleViewPayment = (agreementId) => {
        navigate("/paymentdetails", { state: { agreementId } });
    };

    const AgreementCard = ({ agreement }) => {
        const isPaid = agreement.payment_exists; // This would need to be added to the backend query
        
        return (
            <Box
                borderWidth="1px"
                borderRadius="lg"
                overflow="hidden"
                p="4"
                bg="white"
                boxShadow="md"
                _hover={{ boxShadow: "lg" }}
                transition="all 0.3s">
                
                <Flex direction="column" gap="2">
                    <Flex justify="space-between" align="center">
                        <Text fontWeight="bold" fontSize="lg">
                            {agreement.make} {agreement.model}
                        </Text>
                        <Badge colorScheme={isPaid ? "green" : "orange"}>
                            {isPaid ? "Paid" : "Pending Payment"}
                        </Badge>
                    </Flex>
                    
                    <Text fontSize="sm" color="gray.600">
                        Year: {agreement.p_year}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Color: {agreement.color}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Dealership: {agreement.dealership_name}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Agreement ID: {agreement.a_id}
                    </Text>
                    
                    <Text fontSize="sm" color="gray.600">
                        Agreement Date: {new Date(agreement.datum).toLocaleDateString()}
                    </Text>
                    
                    <Text fontSize="lg" fontWeight="bold" color="blue.600">
                        Price: ${agreement.price}
                    </Text>
                    
                    {!isPaid ? (
                        <Button
                            colorScheme="green"
                            size="sm"
                            onClick={() => handleMakePayment(agreement)}
                            marginTop="2">
                            Make Payment
                        </Button>
                    ) : (
                        <Button
                            colorScheme="blue"
                            size="sm"
                            onClick={() => handleViewPayment(agreement.a_id)}
                            marginTop="2">
                            View Payment
                        </Button>
                    )}
                </Flex>
            </Box>
        );
    };

    if (loading) {
        return (
            <Flex justify="center" align="center" height="200px">
                <Text>Loading agreements...</Text>
            </Flex>
        );
    }

    return (
        <Flex width='100%' flexDir='column' justifyContent='center'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                My Agreements
            </Text>
            
            {agreements.length === 0 ? (
                <Flex justify="center" align="center" height="200px">
                    <Text fontSize="lg" color="gray.500">
                        No agreements found. Agreements will appear here when dealerships create them for you.
                    </Text>
                </Flex>
            ) : (
                <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                    {agreements.map((agreement, index) => (
                        <AgreementCard key={agreement.a_id || index} agreement={agreement} />
                    ))}
                </SimpleGrid>
            )}
        </Flex>
    );
};

export default MyAgreements;
