import { Formik, Form, Field } from "formik";
import { Button, Flex, Text, useToast, Box } from "@chakra-ui/react";
import { Input, InputGroup, InputLeftElement } from "@chakra-ui/react";
import { EmailIcon, LockIcon, PhoneIcon, InfoIcon } from "@chakra-ui/icons";
import { useNavigate } from "react-router-dom";
import * as Yup from "yup";

const SignupSchema = Yup.object().shape({
    EMBG: Yup.string()
        .min(13, "Must be 13 digits exactly!")
        .max(13, "Must be 13 digits exactly!")
        .required("This field is required!")
        .matches(/^[0-9]*$/, "Must be a number."),
    C_name: Yup.string()
        .min(5, "Too Short!")
        .max(50, "Too Long!")
        .required("This field is required!"),

    Email: Yup.string()
        .email("Invalid email")
        .required("This field is required!"),
    Pass: Yup.string()
        .required("Must enter a password!")
        .min(8, "Must have 8 or more characters.")
        .matches(
            /^(?=.*[a-z])(?=.*[A-Z])(?=.*[0-9])(?=.*[!@#\$%\^&\*])(?=.{8,})/,
            "Must have one uppercase, one lowercase, one number and one special case character."
        ),
    Telephone: Yup.string()
        .min(9, "Phone number must be 9 digits exactly.")
        .max(9, "Phone number must be 9 digits exactly.")
        .matches(/^[0-9]*$/, "Must be a number."),
});

const RegisterForm = () => {
    const navigate = useNavigate();
    const toast = useToast();

    const handleSubmit = async (values, { setSubmitting }) => {
        try {
            const response = await fetch("http://localhost:3001/registeruser", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                body: JSON.stringify(values),
            });

            if (response.ok) {
                toast({
                    title: "Registration Successful",
                    description: "Your client account has been created successfully!",
                    status: "success",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/login");
            } else {
                const errorData = await response.json();
                throw new Error(errorData.error || "Registration failed");
            }
        } catch (error) {
            toast({
                title: "Registration Failed",
                description: error.message,
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setSubmitting(false);
        }
    };

    return (
        <Flex width='100%' flexDir='column' justifyContent='center' maxWidth='500px'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                Register as Client
            </Text>
            
            <Formik
                initialValues={{
                    EMBG: "",
                    C_name: "",
                    Email: "",
                    Pass: "",
                    Telephone: "",
                }}
                validationSchema={SignupSchema}
                onSubmit={handleSubmit}>
                {({ isSubmitting, errors, touched }) => (
                    <Form>
                        <Flex flexDir='column' gap='4'>
                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    EMBG (Personal ID) *
                                </Text>
                                <Field name="EMBG">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="1507995410010"
                                                isInvalid={errors.EMBG && touched.EMBG}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.EMBG && touched.EMBG && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.EMBG}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Full Name *
                                </Text>
                                <Field name="C_name">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="Jane Doe"
                                                isInvalid={errors.C_name && touched.C_name}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.C_name && touched.C_name && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.C_name}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Email *
                                </Text>
                                <Field name="Email">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <EmailIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                type="email"
                                                placeholder="jane@acme.com"
                                                isInvalid={errors.Email && touched.Email}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.Email && touched.Email && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.Email}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Password *
                                </Text>
                                <Field name="Pass">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <LockIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                type="password"
                                                placeholder="Enter secure password"
                                                isInvalid={errors.Pass && touched.Pass}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.Pass && touched.Pass && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.Pass}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Telephone *
                                </Text>
                                <Field name="Telephone">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <PhoneIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="077461321"
                                                isInvalid={errors.Telephone && touched.Telephone}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.Telephone && touched.Telephone && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.Telephone}
                                    </Text>
                                )}
                            </Flex>

                            <Button
                                type="submit"
                                isLoading={isSubmitting}
                                loadingText="Registering..."
                                colorScheme="blue"
                                size="lg"
                                marginTop="4">
                                Register Client
                            </Button>
                        </Flex>
                    </Form>
                )}
            </Formik>
        </Flex>
    );
};

export default RegisterForm;
