import { Formik, Form, Field } from "formik";
import { Button, Flex, Text, useToast, Box, Select } from "@chakra-ui/react";
import { Input, InputGroup, InputLeftElement } from "@chakra-ui/react";
import { InfoIcon } from "@chakra-ui/icons";
import { useNavigate } from "react-router-dom";
import * as Yup from "yup";

const SignupSchema = Yup.object().shape({
    vin: Yup.string()
        .required("This field is required!")
        .min(5, "VINs are longer")
        .max(20, "VINs are shorter"),
    car_body: Yup.string().required("This field is required!"),
    car_make: Yup.string().required("This field is required!"),
    car_model: Yup.string().required("This field is required!"),
    p_year: Yup.string()
        .required("This field is required!")
        .min(4, "Invaild Year.")
        .max(4, "Invalid Year")
        .matches(/^[0-9]*$/, "Must be a number."),
    capacity: Yup.string().required("This field is required!"),
    power: Yup.string().required("This field is required!"),
    color: Yup.string().required("This field is required!"),
    price: Yup.string()
        .required("This field is required!")
        .min(4, "Invalid Price")
        .max(6, "Invalid Price")
        .matches(/^[0-9]*$/, "Must be a number."),
});

const AddVehicle = () => {
    const navigate = useNavigate();
    const toast = useToast();

    const handleSubmit = async (values, { setSubmitting }) => {
        try {
            const response = await fetch("http://localhost:3001/createvehicle", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                credentials: 'include',
                body: JSON.stringify(values),
            });

            if (response.ok) {
                toast({
                    title: "Vehicle Added Successfully",
                    description: "Your vehicle has been added to the listings!",
                    status: "success",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/mylist");
            } else if (response.status === 401) {
                toast({
                    title: "Unauthorized",
                    description: "Please log in as a dealership to add vehicles",
                    status: "error",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/login");
            } else {
                throw new Error("Failed to add vehicle");
            }
        } catch (error) {
            toast({
                title: "Error",
                description: "Failed to add vehicle. Please try again.",
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setSubmitting(false);
        }
    };

    return (
        <Flex width='100%' flexDir='column' justifyContent='center' maxWidth='500px'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                Add New Vehicle
            </Text>
            
            <Formik
                initialValues={{
                    vin: "",
                    car_body: "",
                    car_make: "",
                    car_model: "",
                    p_year: "",
                    capacity: "",
                    power: "",
                    color: "",
                    price: ""
                }}
                validationSchema={SignupSchema}
                onSubmit={handleSubmit}>
                {({ isSubmitting, errors, touched }) => (
                    <Form>
                        <Flex flexDir='column' gap='4'>
                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    VIN (Vehicle Identification Number) *
                                </Text>
                                <Field name="vin">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="WUZZ1255643"
                                                isInvalid={errors.vin && touched.vin}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.vin && touched.vin && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.vin}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Body Type *
                                </Text>
                                <Field name="car_body">
                                    {({ field }) => (
                                        <Select
                                            {...field}
                                            placeholder="Select body type"
                                            isInvalid={errors.car_body && touched.car_body}>
                                            <option value='Car'>Car</option>
                                            <option value='Motorcycle'>Motorcycle</option>
                                            <option value='Truck'>Truck</option>
                                            <option value='Van'>Van</option>
                                        </Select>
                                    )}
                                </Field>
                                {errors.car_body && touched.car_body && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.car_body}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Make *
                                </Text>
                                <Field name="car_make">
                                    {({ field }) => (
                                        <Select
                                            {...field}
                                            placeholder="Select make"
                                            isInvalid={errors.car_make && touched.car_make}>
                                            <option value='BMW'>BMW</option>
                                            <option value='MINI'>MINI</option>
                                            <option value='Mercedes'>Mercedes</option>
                                            <option value='Porsche'>Porsche</option>
                                            <option value='VW'>VW</option>
                                            <option value='Scania'>Scania</option>
                                        </Select>
                                    )}
                                </Field>
                                {errors.car_make && touched.car_make && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.car_make}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Model *
                                </Text>
                                <Field name="car_model">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="G30 530d"
                                                isInvalid={errors.car_model && touched.car_model}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.car_model && touched.car_model && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.car_model}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Year *
                                </Text>
                                <Field name="p_year">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="2017"
                                                isInvalid={errors.p_year && touched.p_year}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.p_year && touched.p_year && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.p_year}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Engine Capacity *
                                </Text>
                                <Field name="capacity">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="3000cc"
                                                isInvalid={errors.capacity && touched.capacity}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.capacity && touched.capacity && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.capacity}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Power (kW) *
                                </Text>
                                <Field name="power">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="150"
                                                isInvalid={errors.power && touched.power}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.power && touched.power && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.power}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Color *
                                </Text>
                                <Field name="color">
                                    {({ field }) => (
                                        <Select
                                            {...field}
                                            placeholder="Select color"
                                            isInvalid={errors.color && touched.color}>
                                            <option value='red'>Red</option>
                                            <option value='green'>Green</option>
                                            <option value='blue'>Blue</option>
                                            <option value='black'>Black</option>
                                            <option value='white'>White</option>
                                            <option value='yellow'>Yellow</option>
                                            <option value='orange'>Orange</option>
                                            <option value='purple'>Purple</option>
                                            <option value='silver'>Silver</option>
                                            <option value='grey'>Grey</option>
                                            <option value='cyan'>Cyan</option>
                                            <option value='magenta'>Magenta</option>
                                        </Select>
                                    )}
                                </Field>
                                {errors.color && touched.color && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.color}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Price ($) *
                                </Text>
                                <Field name="price">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="35600"
                                                isInvalid={errors.price && touched.price}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.price && touched.price && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.price}
                                    </Text>
                                )}
                            </Flex>

                            <Button
                                type="submit"
                                isLoading={isSubmitting}
                                loadingText="Adding Vehicle..."
                                colorScheme="blue"
                                size="lg"
                                marginTop="4">
                                Add Vehicle
                            </Button>
                        </Flex>
                    </Form>
                )}
            </Formik>
        </Flex>
    );
};

export default AddVehicle;
