import { Formik, Form, Field } from "formik";
import { Button, Flex, Text, useToast } from "@chakra-ui/react";
import { Input, InputGroup, InputLeftElement } from "@chakra-ui/react";
import { EmailIcon, LockIcon, PhoneIcon, InfoIcon } from "@chakra-ui/icons";
import { useNavigate } from "react-router-dom";
import * as Yup from "yup";

const DealershipRegisterForm = () => {
    const navigate = useNavigate();
    const toast = useToast();

    const validationSchema = Yup.object({
        tax_nr: Yup.string()
            .required("Tax Number is required")
            .min(5, "Tax Number must be at least 5 characters"),
        d_name: Yup.string()
            .required("Dealership Name is required")
            .min(2, "Dealership Name must be at least 2 characters"),
        brands: Yup.string()
            .required("Brands are required")
            .min(2, "Please specify at least one brand"),
        email: Yup.string()
            .email("Invalid email format")
            .required("Email is required"),
        pass: Yup.string()
            .required("Password is required")
            .min(6, "Password must be at least 6 characters"),
        director: Yup.string()
            .required("Director name is required")
            .min(2, "Director name must be at least 2 characters"),
        telephones: Yup.string()
            .required("Telephone number is required")
            .min(8, "Please enter a valid telephone number"),
        addres: Yup.string()
            .required("Address is required")
            .min(5, "Please enter a complete address"),
    });

    const initialValues = {
        tax_nr: "",
        d_name: "",
        brands: "",
        email: "",
        pass: "",
        director: "",
        telephones: "",
        addres: "",
    };

    const handleSubmit = async (values, { setSubmitting }) => {
        try {
            // Convert brands string to array (comma-separated)
            const brandsArray = values.brands.split(',').map(brand => brand.trim());
            const telephonesArray = values.telephones.split(',').map(tel => tel.trim());

            const requestData = {
                ...values,
                brands: brandsArray,
                telephones: telephonesArray,
            };

            const response = await fetch("http://localhost:3001/registerdealership", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                body: JSON.stringify(requestData),
            });

            if (response.ok) {
                toast({
                    title: "Registration Successful",
                    description: "Your dealership has been registered successfully!",
                    status: "success",
                    duration: 3000,
                    isClosable: true,
                });
                navigate("/dealershiplogin");
            } else {
                const errorData = await response.json();
                throw new Error(errorData.error || "Registration failed");
            }
        } catch (error) {
            toast({
                title: "Registration Failed",
                description: error.message,
                status: "error",
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setSubmitting(false);
        }
    };

    return (
        <Flex width='100%' flexDir='column' justifyContent='center' maxWidth='500px'>
            <Text
                marginBottom='20px'
                textAlign='center'
                fontWeight='bold'
                fontSize='30px'
                justifySelf='center'>
                Register as Dealership
            </Text>
            
            <Formik
                initialValues={initialValues}
                validationSchema={validationSchema}
                onSubmit={handleSubmit}>
                {({ isSubmitting, errors, touched }) => (
                    <Form>
                        <Flex flexDir='column' gap='4'>
                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Tax Number *
                                </Text>
                                <Field name="tax_nr">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="Enter tax number"
                                                isInvalid={errors.tax_nr && touched.tax_nr}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.tax_nr && touched.tax_nr && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.tax_nr}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Dealership Name *
                                </Text>
                                <Field name="d_name">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="Enter dealership name"
                                                isInvalid={errors.d_name && touched.d_name}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.d_name && touched.d_name && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.d_name}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Brands (comma-separated) *
                                </Text>
                                <Field name="brands">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="e.g., BMW, Mercedes, Audi"
                                                isInvalid={errors.brands && touched.brands}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.brands && touched.brands && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.brands}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Email *
                                </Text>
                                <Field name="email">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <EmailIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                type="email"
                                                placeholder="Enter email"
                                                isInvalid={errors.email && touched.email}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.email && touched.email && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.email}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Password *
                                </Text>
                                <Field name="pass">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <LockIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                type="password"
                                                placeholder="Enter password"
                                                isInvalid={errors.pass && touched.pass}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.pass && touched.pass && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.pass}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Director Name *
                                </Text>
                                <Field name="director">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="Enter director name"
                                                isInvalid={errors.director && touched.director}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.director && touched.director && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.director}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Telephone Numbers (comma-separated) *
                                </Text>
                                <Field name="telephones">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <PhoneIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="e.g., +1234567890, +0987654321"
                                                isInvalid={errors.telephones && touched.telephones}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.telephones && touched.telephones && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.telephones}
                                    </Text>
                                )}
                            </Flex>

                            <Flex flexDir='column'>
                                <Text marginBottom='5px' fontWeight='bold'>
                                    Address *
                                </Text>
                                <Field name="addres">
                                    {({ field }) => (
                                        <InputGroup>
                                            <InputLeftElement pointerEvents='none'>
                                                <InfoIcon color='gray.300' />
                                            </InputLeftElement>
                                            <Input
                                                {...field}
                                                placeholder="Enter complete address"
                                                isInvalid={errors.addres && touched.addres}
                                            />
                                        </InputGroup>
                                    )}
                                </Field>
                                {errors.addres && touched.addres && (
                                    <Text color="red.500" fontSize="sm" marginTop="1">
                                        {errors.addres}
                                    </Text>
                                )}
                            </Flex>

                            <Button
                                type="submit"
                                isLoading={isSubmitting}
                                loadingText="Registering..."
                                colorScheme="blue"
                                size="lg"
                                marginTop="4">
                                Register Dealership
                            </Button>
                        </Flex>
                    </Form>
                )}
            </Formik>
        </Flex>
    );
};

export default DealershipRegisterForm;
