package com.example.domify.service.impl;

import com.example.domify.model.*;
import com.example.domify.repository.LeaseRepository;
import com.example.domify.service.*;
//import jakarta.transaction.Transactional;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;
import org.springframework.transaction.annotation.Transactional;


@Service
public class LeaseServiceImpl implements LeaseService {
    private final LeaseRepository leaseRepository;
    private final ListingService listingService;
    private final UserService userService;
    private final TenantProfileService tenantProfileService;
    private final LandlordProfileService landlordProfileService;

    public LeaseServiceImpl(LeaseRepository leaseRepository,
                            ListingService listingService,
                            UserService userService,
                            TenantProfileService tenantProfileService,
                            LandlordProfileService landlordProfileService) {
        this.leaseRepository = leaseRepository;
        this.listingService = listingService;
        this.userService = userService;
        this.tenantProfileService = tenantProfileService;
        this.landlordProfileService = landlordProfileService;
    }

    @Transactional(rollbackFor = Exception.class, isolation = Isolation.READ_COMMITTED)
    public Lease save(Long listingId,
                                        Long tenantId,
                                        Long landlordUserId,
                                        LocalDate startDate,
                                        LocalDate endDate,
                                        BigDecimal rentAmount,
                                        BigDecimal depositAmount) {
        try {
            System.out.println("=== ЗАПОЧНУВАЊЕ НА ТРАНСАКЦИЈА ЗА КРЕИРАЊЕ ДОГОВОР ===");
            Listing listing = listingService.findById(listingId);

            if (!"available".equals(listing.getStatus())) {
                throw new IllegalStateException("Огласот не е достапен за изнајмување. Статус: " + listing.getStatus());
            }

            TenantProfile tenant = tenantProfileService.findByUserId(tenantId);
            if (tenant == null) {
                throw new IllegalArgumentException("Изнајмувачот не е пронајден со ID: " + tenantId);
            }

            LandlordProfile landlord = landlordProfileService.findByUserId(landlordUserId);
            if (landlord == null) {
                throw new IllegalArgumentException("Издавачот не е пронајден со ID: " + landlordUserId);
            }

            System.out.println("TENANT ID: " + tenant.getId());
            System.out.println("LANDLORD ID: " + landlord.getId());

            Lease lease = new Lease(
                    startDate,
                    endDate,
                    rentAmount,
                    depositAmount,
                    listing,
                    tenant,
                    landlord
            );

            Lease savedLease = leaseRepository.save(lease);
            listing.setStatus("изнајмено");
            listingService.save(listing);
            return savedLease;

        } catch (Exception e) {
            System.err.println("ГРЕШКА ВО ТРАНСАКЦИЈАТА: " + e.getMessage());
            throw new RuntimeException("Грешка при креирање на договор за изнајмување: " + e.getMessage(), e);
        }
    }

    public List<Lease> findByLandlord(Long landlordId) {
        return leaseRepository.findByLandlordId(landlordId);
    }

    public List<Lease> findByTenant(Long tenantId) {
        return leaseRepository.findByTenantId(tenantId);
    }

    @Override
    public Lease findById(Long leaseId) {
        return leaseRepository.findById(leaseId).orElseThrow(RuntimeException::new);
    }

    @Transactional
    public void rateUser(Long leaseId, Long raterProfileId, Integer rating, boolean isRatingTenant) {
        Lease lease = leaseRepository.findById(leaseId)
                .orElseThrow(() -> new IllegalArgumentException("Lease not found"));

        boolean isRaterLandlord = lease.getLandlord().getId().equals(raterProfileId);
        boolean isRaterTenant = lease.getTenant().getId().equals(raterProfileId);

        if (!isRaterLandlord && !isRaterTenant) {
            throw new IllegalArgumentException("User is not part of this lease");
        }

        if (isRatingTenant && !isRaterLandlord) {
            throw new IllegalArgumentException("Only landlord can rate tenant");
        }

        if (!isRatingTenant && !isRaterTenant) {
            throw new IllegalArgumentException("Only tenant can rate landlord");
        }

        UserD userToRate = isRatingTenant ? lease.getTenant().getUser() : lease.getLandlord().getUser();

        userService.updateUserRating(userToRate.getId(), BigDecimal.valueOf(rating));
    }
}
