import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpExchange;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.sql.SQLException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.io.File;

public class FrontendHandler implements HttpHandler {
    private static final ExecutorService executorService = Executors.newFixedThreadPool(1);

    @Override
    public void handle(HttpExchange exchange) throws IOException {
        String uri = exchange.getRequestURI().toString();
        String path = exchange.getRequestURI().getPath();
        String method = exchange.getRequestMethod();

        if ("GET".equalsIgnoreCase(method)) {
            handleGetRequests(exchange, path, uri);
        } else if ("POST".equalsIgnoreCase(method)) {
            handlePostRequests(exchange, path);
        } else {
            Server.sendResponse(exchange, 405, "{\"message\": \"Method not allowed.\"}");
        }
    }

    private void handleGetRequests(HttpExchange exchange, String path, String uri) throws IOException {
        switch (path) {
            case "/admin/last-update-time":
                String lastUpdateTimeResponse = "{\"lastUpdateTime\": \"" + Server.getLastUpdateTime().toString() + "\"}";
                Server.sendResponse(exchange, 200, lastUpdateTimeResponse);
                break;
            case "/admin/current-options-count":
                int currentOptionsCount;
                try {
                    currentOptionsCount = DatabaseUtil.getCurrentOptionsCount();
                } catch (SQLException e) {
                    throw new RuntimeException(e);
                }
                String currentOptionsCountResponse = "{\"currentOptionsCount\": " + currentOptionsCount + "}";
                Server.sendResponse(exchange, 200, currentOptionsCountResponse);
                break;
            case "/admin/changed-options-count":
                int changedOptionsCount;
                try {
                    changedOptionsCount = DatabaseUtil.getChangedOptionsCountSinceLastUpdate();
                } catch (SQLException e) {
                    throw new RuntimeException(e);
                }
                String changedOptionsCountResponse = "{\"changedOptionsCount\": " + changedOptionsCount + "}";
                Server.sendResponse(exchange, 200, changedOptionsCountResponse);
                break;
            case"/admin/drop-db":
                try {
                    System.out.println("DROPPING DATABASE!");
                    DatabaseUtil.dropOptions();
                } catch (SQLException e) {
                    throw new RuntimeException(e);
                }
                break;
            default:
                serveStaticFiles(exchange, uri);
                break;
        }
    }

    private void handlePostRequests(HttpExchange exchange, String path) throws IOException {
        if (path.equalsIgnoreCase("/admin/start-scraper")) {
            Future<Void> future = executorService.submit(new Scraper());
            Server.updateLastUpdateTime();
            Server.sendResponse(exchange, 200, "{\"message\": \"Starting options update.\"}");
            executorService.submit(() -> {
                try {
                    future.get();
                } catch (Exception e) {
                    e.printStackTrace();

                }
            });
        } else if (path.equalsIgnoreCase("/submit")) {
            handleFormSubmit(exchange);
        }else {
            Server.sendResponse(exchange, 404, "{\"message\": \"Endpoint not found.\"}");
        }
    }

    private void handleFormSubmit(HttpExchange exchange) throws IOException {
        String requestBody = new String(exchange.getRequestBody().readAllBytes());
        System.out.println("Form data: " + requestBody);
        ObjectMapper mapper = new ObjectMapper();
        Map<String, String> formData = mapper.readValue(requestBody, Map.class);
        String destination = formData.get("destination");
        String departureDate = formData.get("departureDate");
        String nightsNumberStr = formData.get("nightsNumber");
        String numPeopleStr = formData.get("numberPeople");
        int numberOfNights = (nightsNumberStr != null && !nightsNumberStr.isEmpty()) ? Integer.parseInt(nightsNumberStr) : 0;
        int numPeople = Integer.parseInt(numPeopleStr);
        String queryDate = "";
        boolean dateFlag = false;

        if (departureDate != null && !departureDate.isEmpty()) {
            if (numberOfNights != 0) {
                queryDate = LocalDate.parse(departureDate, DateTimeFormatter.ofPattern("yyyy-MM-dd"))
                        .format(DateTimeFormatter.ofPattern("dd.MM.yyyy"));
                String dateTo = LocalDate.parse(departureDate, DateTimeFormatter.ofPattern("yyyy-MM-dd"))
                        .plusDays(numberOfNights).format(DateTimeFormatter.ofPattern("dd.MM.yyyy"));
                queryDate += " - " + dateTo;
            } else {
                dateFlag = true; // send only from date
                queryDate = LocalDate.parse(departureDate, DateTimeFormatter.ofPattern("yyyy-MM-dd"))
                        .format(DateTimeFormatter.ofPattern("dd.MM.yyyy"));
            }
        }

        List<Option> options;
        try {
            options = DatabaseUtil.queryOptions(destination, queryDate, numPeople, dateFlag);
        } catch (SQLException e) {
            e.printStackTrace();
            exchange.sendResponseHeaders(500, -1);
            exchange.close();
            return;
        }
        Server.sendResponse(exchange, 200, mapper.writeValueAsString(options));
    }

    private void serveStaticFiles(HttpExchange exchange, String uri) throws IOException {
        File file;
        String contentType = "text/html";

        if (uri.equals("/admin-panel")) {
            System.out.println("Serving admin panel");
            file = new File(System.getProperty("user.dir") + "/frontend/admin-panel.html");
        } else {
            file = new File(System.getProperty("user.dir") + "/frontend" + uri);
            if (uri.endsWith(".css")) {
                contentType = "text/css";
            } else if (uri.endsWith(".js")) {
                contentType = "application/javascript";
            }
        }

        if (!file.exists() || !file.isFile()) {
            file = new File("frontend/index.html");
        }

        if (file.exists() && file.isFile()) {
            exchange.getResponseHeaders().set("Content-Type", contentType);
            exchange.sendResponseHeaders(200, file.length());
            OutputStream outputStream = exchange.getResponseBody();
            Files.copy(file.toPath(), outputStream);
            outputStream.close();
        } else {
            exchange.sendResponseHeaders(404, -1);
        }
    }

}
