from subjects.consts import SUBJECTS_VECTOR
from subjects.management.commands import subjects_vector
from subjects.models import Subject
from pathlib import Path
import json
from numpy import average

def get_eligible_subjects(student):
    passed_ids = set(student.passed_subjects.values_list('id', flat=True))

    total_credits = student.total_credits
    level_credits = student.level_credits
    study_track = student.study_track

    all_subjects = (Subject.objects
        .exclude(id__in=passed_ids)
        .select_related('subject_info')
    )

    if level_credits[0] >= 6:
        all_subjects = all_subjects.exclude(subject_info__level=1)
    if level_credits[1] >= 36:
        all_subjects = all_subjects.exclude(subject_info__level=2)

    valid_subjects = []
    for subject in all_subjects:
        subject_info_ = subject.subject_info
        prereqs = subject_info_.prerequisite or {}
        if prereqs.get('credits') and total_credits < prereqs['credits']:
            continue
        if prereqs.get('subjects') and not any(subj_id in passed_ids for subj_id in prereqs['subjects']):
            continue
        if study_track not in subject_info_.elective_for:
            continue
        valid_subjects.append(subject)

    return valid_subjects

def student_vector(student):
    base_dir = Path(__file__).resolve().parent
    vocab_file_path = base_dir / 'management' / 'data' / 'vocabulary.json'
    try:
        with open(vocab_file_path, 'r', encoding='utf-8') as f:
            vocabulary = json.load(f)
    except FileNotFoundError:
        print("file not found")
        return -1

    student_vector = {}
    student_vector['index'] = student.index
    for key in vocabulary:
        if key == "assistants": continue
        student_values = getattr(student, key, [])

        student_vector[key] = []
        words = vocabulary[key]
        for word in words:
            student_vector[key].append(0 if word not in student_values else 1)

    student_vector['study_effort'] = student.study_effort / 5
    student_vector['current_year'] = student.current_year

    return student_vector


def map_to_subjects_vector(subjects):
    filtered_subject_vectors = {}
    for subject in subjects:
        vector = SUBJECTS_VECTOR.get(subject.name)
        if vector:
            filtered_subject_vectors[subject.name] = vector
    
    return filtered_subject_vectors
    

def score_for_preferences(student_vector, eligible_subjects):
    K = 1
    filtered_subjects_vector = {}
    for subject in eligible_subjects:
        filtered_subjects_vector[subject] = {}
        values = eligible_subjects[subject]
        for key in student_vector:
            if key in ["index", "study_effort", "current_year"]: continue
            student_values = student_vector[key]
            subject_values = values[key]
            tot_count = 0
            match_count = 0
            if key == "tags" and subject == "Дигитална форензика":
                print(student_values)
                print(subject_values)

            for i in range(len(student_values)):
                if student_values[i] == 1:
                    tot_count += 1
                    if subject_values[i] == 1:
                        match_count += 1
            
            # score = (match_count + K) / (tot_count + K * len(student_values))
            score = match_count / tot_count if tot_count != 0 else 0
            filtered_subjects_vector[subject][key] = score
        
        study_effort = student_vector["study_effort"]
        # current_year = student_vector["current_year"]

        # TODO: add something for people who want to study more!
        filtered_subjects_vector[subject]['easiness'] = (1 - study_effort) * values['isEasy']
        filtered_subjects_vector[subject]['activated'] = values['activated']
        filtered_subjects_vector[subject]['participant_score'] = values['participants']

    return filtered_subjects_vector

WEIGHTS = {
    "professors": 0.05,
    "technologies": 0.05,
    "tags": 0.4,
    "evaluation": 0.1, 
    "easiness": 0.3,
    "activated": 1,
    "participant_score": 0.1,
}

def get_recommendations(filtered_subjects_vector):
    subject_scores = {}
    max_ = -1
    for subject in filtered_subjects_vector:
        keys = filtered_subjects_vector[subject]
        score = 0
        for key in keys:
            score += WEIGHTS[key] * keys[key]
        
        max_ = max(score, max_)
        subject_scores[subject] = score
    
    if max_ == 0: return filtered_subjects_vector.keys()
    for subject in subject_scores:
        subject_scores[subject] /= max_
    

    N = 7
    top_subjects = dict(sorted(subject_scores.items(), key=lambda item: item[1], reverse=True))
    return top_subjects