#!/bin/bash

# Export Docker database and import to local
echo "🔄 Copying Docker database to local machine..."

# Configuration
LOCAL_DB_NAME="kupi_mk"
LOCAL_DB_USER="admin"
DOCKER_CONTAINER="kupi-mk-postgres"
EXPORT_FILE="/tmp/docker-db-export.sql"

# Step 1: Check if Docker container is running
echo "🐳 Step 1: Checking Docker container..."
if ! docker ps | grep -q $DOCKER_CONTAINER; then
    echo "❌ Docker container '$DOCKER_CONTAINER' is not running!"
    echo "Start it with: docker-compose up -d postgres"
    exit 1
fi

echo "✅ Docker container is running!"

# Step 2: Export from Docker database
echo "📤 Step 2: Exporting from Docker database..."
docker exec $DOCKER_CONTAINER pg_dump -U admin -d kupi_mk \
  --no-owner --no-acl --clean --if-exists > "$EXPORT_FILE"

if [ $? -ne 0 ]; then
    echo "❌ Failed to export Docker database!"
    exit 1
fi

echo "✅ Docker database exported successfully!"
echo "📁 Export size: $(du -h "$EXPORT_FILE" | cut -f1)"

# Step 3: Import to local database
echo "📥 Step 3: Importing to local database..."
psql -h localhost -p 5432 -U $LOCAL_DB_USER -d $LOCAL_DB_NAME -f "$EXPORT_FILE"

if [ $? -eq 0 ]; then
    echo "✅ Database successfully copied to local machine!"
    echo ""
    echo "🎉 Your Docker database is now available locally!"
    echo "   Local database: localhost:5432"
    echo ""
    echo "🔍 To verify, you can check with:"
    echo "   psql -h localhost -p 5432 -U $LOCAL_DB_USER -d $LOCAL_DB_NAME -c '\\dt'"
    
    # Cleanup
    rm "$EXPORT_FILE"
    
else
    echo "❌ Failed to import database to local machine!"
fi