#!/bin/bash

# Kupi.mk Docker Setup Script with Port Conflict Resolution
set -e

echo "🚀 Setting up Kupi.mk with Docker (with port conflict resolution)..."

# Function to check if port is in use
check_port() {
    local port=$1
    if lsof -Pi :$port -sTCP:LISTEN -t >/dev/null 2>&1; then
        return 0 # Port is in use
    else
        return 1 # Port is free
    fi
}

# Function to find next available port
find_available_port() {
    local start_port=$1
    local port=$start_port
    while check_port $port; do
        ((port++))
        if [ $port -gt $((start_port + 100)) ]; then
            echo "❌ Could not find available port after checking 100 ports starting from $start_port"
            exit 1
        fi
    done
    echo $port
}

# Check and resolve port conflicts
echo "🔍 Checking for port conflicts..."

# Check PostgreSQL port (5432)
if check_port 5432; then
    echo "⚠️  Port 5432 is in use (probably by existing PostgreSQL)"
    echo "   Using port 5433 for Docker PostgreSQL instead"
    DB_PORT=5433
else
    echo "✅ Port 5432 is available"
    DB_PORT=5432
fi

# Check Backend port (5001)
if check_port 5001; then
    BACKEND_PORT=$(find_available_port 5002)
    echo "⚠️  Port 5001 is in use, using port $BACKEND_PORT instead"
else
    echo "✅ Port 5001 is available"
    BACKEND_PORT=5001
fi

# Check Frontend port (3000)
if check_port 3000; then
    FRONTEND_PORT=$(find_available_port 3001)
    echo "⚠️  Port 3000 is in use, using port $FRONTEND_PORT instead"
else
    echo "✅ Port 3000 is available"
    FRONTEND_PORT=3000
fi

# Create dynamic docker-compose.yml with available ports
echo "📝 Creating docker-compose.yml with available ports..."
cat > docker-compose.dynamic.yml << EOF
version: '3.8'

services:
  postgres:
    image: postgres:15-alpine
    container_name: kupi-mk-postgres
    environment:
      POSTGRES_DB: kupi_mk
      POSTGRES_USER: admin
      POSTGRES_PASSWORD: password123
    ports:
      - "$DB_PORT:5432"
    volumes:
      - postgres_data:/var/lib/postgresql/data
      - ./database_setup.sql:/docker-entrypoint-initdb.d/01-setup.sql
      - ./database_updates_cart_reviews.sql:/docker-entrypoint-initdb.d/02-updates.sql
    healthcheck:
      test: ["CMD-SHELL", "pg_isready -U admin -d kupi_mk"]
      interval: 10s
      timeout: 5s
      retries: 5
    networks:
      - kupi-mk-network

  backend:
    build: 
      context: ./backend
      dockerfile: Dockerfile
    container_name: kupi-mk-backend
    environment:
      NODE_ENV: production
      PORT: 5001
      DB_HOST: postgres
      DB_PORT: 5432
      DB_NAME: kupi_mk
      DB_USER: admin
      DB_PASSWORD: password123
      JWT_SECRET: your-super-secret-jwt-key-change-this-in-production
    ports:
      - "$BACKEND_PORT:5001"
    depends_on:
      postgres:
        condition: service_healthy
    volumes:
      - backend_uploads:/app/uploads
    networks:
      - kupi-mk-network
    restart: unless-stopped

  frontend:
    build:
      context: ./frontend
      dockerfile: Dockerfile
    container_name: kupi-mk-frontend
    ports:
      - "$FRONTEND_PORT:80"
    depends_on:
      - backend
    networks:
      - kupi-mk-network
    restart: unless-stopped

volumes:
  postgres_data:
  backend_uploads:

networks:
  kupi-mk-network:
    driver: bridge
EOF

# Check Docker installation
if ! command -v docker &> /dev/null; then
    echo "❌ Docker is not installed. Please install Docker first."
    exit 1
fi

if ! command -v docker-compose &> /dev/null; then
    echo "❌ Docker Compose is not installed. Please install Docker Compose first."
    exit 1
fi

# Create environment files if they don't exist
if [ ! -f "backend/.env" ]; then
    echo "📝 Creating backend .env file..."
    cp backend/.env.docker backend/.env
fi

if [ ! -f "frontend/.env" ]; then
    echo "📝 Creating frontend .env file..."
    # Update frontend .env with correct backend URL
    cat > frontend/.env << EOF
REACT_APP_API_URL=http://localhost:$BACKEND_PORT/api
REACT_APP_BACKEND_URL=http://localhost:$BACKEND_PORT
GENERATE_SOURCEMAP=false
EOF
fi

# Clean up any existing containers
echo "🧹 Cleaning up existing containers..."
docker-compose -f docker-compose.dynamic.yml down --remove-orphans 2>/dev/null || true
docker rm -f kupi-mk-postgres kupi-mk-backend kupi-mk-frontend 2>/dev/null || true

# Build and start containers
echo "🏗️  Building and starting containers..."
docker-compose -f docker-compose.dynamic.yml up --build -d

# Wait for services to be ready
echo "⏳ Waiting for services to be ready..."
sleep 30

# Check if services are running
if docker-compose -f docker-compose.dynamic.yml ps | grep -q "Up"; then
    echo "✅ Services are running!"
    echo ""
    echo "🌐 Application URLs:"
    echo "   Frontend: http://localhost:$FRONTEND_PORT"
    echo "   Backend API: http://localhost:$BACKEND_PORT"
    echo "   Database: localhost:$DB_PORT"
    echo ""
    echo "📊 To view logs:"
    echo "   docker-compose -f docker-compose.dynamic.yml logs -f"
    echo ""
    echo "🛑 To stop:"
    echo "   docker-compose -f docker-compose.dynamic.yml down"
    echo ""
    echo "💡 The dynamic compose file 'docker-compose.dynamic.yml' has been created with your available ports"
else
    echo "❌ Some services failed to start. Check logs with:"
    echo "   docker-compose -f docker-compose.dynamic.yml logs"
fi