import React, { createContext, useContext, useState } from 'react';
import api from '../services/api';

const ProductContext = createContext();

export const useProducts = () => {
  const context = useContext(ProductContext);
  if (!context) {
    throw new Error('useProducts must be used within a ProductProvider');
  }
  return context;
};

export const ProductProvider = ({ children }) => {
  const [products, setProducts] = useState([]);
  const [categories, setCategories] = useState([]);
  const [loading, setLoading] = useState(false);
  const [pagination, setPagination] = useState({});

  const fetchProducts = async (filters = {}) => {
    setLoading(true);
    try {
      const params = new URLSearchParams();
      Object.keys(filters).forEach(key => {
        if (filters[key]) {
          params.append(key, filters[key]);
        }
      });

      const response = await api.get(`/products?${params}`);
      setProducts(response.data.products);
      setPagination(response.data.pagination);
      return { success: true, data: response.data };
    } catch (error) {
      console.error('Error fetching products:', error);
      return { success: false, message: 'Failed to fetch products' };
    } finally {
      setLoading(false);
    }
  };

  const fetchProduct = async (id) => {
    try {
      const response = await api.get(`/products/${id}`);
      return { success: true, data: response.data };
    } catch (error) {
      return { 
        success: false, 
        message: error.response?.data?.message || 'Product not found' 
      };
    }
  };

  const createProduct = async (productData) => {
    try {
      const formData = new FormData();
      
      // Append regular fields
      Object.keys(productData).forEach(key => {
        if (key !== 'images') {
          formData.append(key, productData[key]);
        }
      });

      // Append images
      if (productData.images) {
        productData.images.forEach(image => {
          formData.append('images', image);
        });
      }

      const response = await api.post('/products', formData, {
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      });

      return { success: true, data: response.data };
    } catch (error) {
      return { 
        success: false, 
        message: error.response?.data?.message || 'Failed to create product',
        errors: error.response?.data?.errors || []
      };
    }
  };

  const fetchCategories = async () => {
    try {
      const response = await api.get('/products/categories/all');
      setCategories(response.data);
      return { success: true, data: response.data };
    } catch (error) {
      return { success: false, message: 'Failed to fetch categories' };
    }
  };

  const value = {
    products,
    categories,
    loading,
    pagination,
    fetchProducts,
    fetchProduct,
    createProduct,
    fetchCategories
  };

  return (
    <ProductContext.Provider value={value}>
      {children}
    </ProductContext.Provider>
  );
};
