import React from 'react';
import { Link } from 'react-router-dom';

const ProductCard = ({ product }) => {
  const getImageUrl = (imagePath) => {
    if (!imagePath) return '/placeholder-image.svg';
    
    // If it's already a full URL, return as is
    if (imagePath.startsWith('http')) return imagePath;
    
    // If it starts with /, add the backend URL
    if (imagePath.startsWith('/')) {
      return `http://localhost:5000${imagePath}`;
    }
    
    // Otherwise assume it's a relative path
    return `http://localhost:5000/uploads/products/${imagePath}`;
  };

  const imageUrl = product.images && product.images.length > 0 
    ? getImageUrl(product.images[0])
    : '/placeholder-image.svg';

  console.log('Product images:', product.images); // Debug log
  console.log('Image URL:', imageUrl); // Debug log

  return (
    <Link to={`/product/${product.id}`} className="group">
      <div className="bg-white rounded-lg shadow-md overflow-hidden hover:shadow-lg transition-shadow duration-300">
        <div className="aspect-w-16 aspect-h-9 bg-gray-200">
          <img
            src={imageUrl}
            alt={product.title}
            className="w-full h-48 object-cover group-hover:scale-105 transition-transform duration-300"
            onError={(e) => {
              console.error('Image failed to load:', e.target.src);
              e.target.src = '/placeholder-image.svg';
            }}
            onLoad={() => {
              console.log('Image loaded successfully:', imageUrl);
            }}
          />
        </div>
        
        <div className="p-4">
          <h3 className="text-lg font-semibold text-gray-900 mb-2 line-clamp-2">
            {product.title}
          </h3>
          
          <p className="text-gray-600 text-sm mb-3 line-clamp-2">
            {product.description}
          </p>
          
          <div className="flex justify-between items-center">
            <span className="text-2xl font-bold text-primary-600">
              {parseFloat(product.price).toLocaleString('mk-MK')} ден.
            </span>
            
            {product.location && (
              <span className="text-xs text-gray-500 bg-gray-100 px-2 py-1 rounded">
                📍 {product.location}
              </span>
            )}
          </div>
          
          <div className="mt-3 flex justify-between items-center text-sm text-gray-500">
            <span>{product.category_name}</span>
            <span>од {product.seller_name}</span>
          </div>
        </div>
      </div>
    </Link>
  );
};

export default ProductCard;
