import React, { useState, useEffect } from 'react';
import { useParams } from 'react-router-dom';
import { useProducts } from '../context/ProductContext';

const ProductDetail = () => {
  const { id } = useParams();
  const { fetchProduct } = useProducts();
  const [product, setProduct] = useState(null);
  const [loading, setLoading] = useState(true);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);

  useEffect(() => {
    const loadProduct = async () => {
      const result = await fetchProduct(id);
      if (result.success) {
        setProduct(result.data);
      }
      setLoading(false);
    };

    loadProduct();
  }, [id, fetchProduct]);

  if (loading) {
    return (
      <div className="flex justify-center items-center min-h-screen">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-primary-600"></div>
      </div>
    );
  }

  if (!product) {
    return (
      <div className="text-center py-12">
        <h2 className="text-2xl font-bold text-gray-900 mb-4">Производот не е пронајден</h2>
        <p className="text-gray-600">Производот што го барате не постои или е отстранет.</p>
      </div>
    );
  }

  const images = product.images && product.images.length > 0 
    ? product.images.map(img => {
        if (img.startsWith('http')) return img;
        return img.startsWith('/') ? `http://localhost:5000${img}` : `http://localhost:5000/uploads/products/${img}`;
      })
    : ['/placeholder-image.svg'];

  return (
    <div className="max-w-6xl mx-auto">
      <div className="bg-white rounded-lg shadow-lg overflow-hidden">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 p-6">
          {/* Image Gallery */}
          <div className="space-y-4">
            <div className="aspect-w-16 aspect-h-12 bg-gray-200 rounded-lg overflow-hidden">
              <img
                src={images[currentImageIndex]}
                alt={product.title}
                className="w-full h-96 object-cover"
                onError={(e) => {
                  e.target.src = '/placeholder-image.jpg';
                }}
              />
            </div>
            
            {images.length > 1 && (
              <div className="grid grid-cols-4 gap-2">
                {images.map((image, index) => (
                  <button
                    key={index}
                    onClick={() => setCurrentImageIndex(index)}
                    className={`aspect-w-1 aspect-h-1 rounded-lg overflow-hidden border-2 ${
                      currentImageIndex === index ? 'border-primary-600' : 'border-transparent'
                    }`}
                  >
                    <img
                      src={image}
                      alt={`${product.title} ${index + 1}`}
                      className="w-full h-20 object-cover"
                    />
                  </button>
                ))}
              </div>
            )}
          </div>

          {/* Product Info */}
          <div className="space-y-6">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">
                {product.title}
              </h1>
              
              <div className="flex items-center space-x-4 text-sm text-gray-600 mb-4">
                <span className="bg-primary-100 text-primary-800 px-3 py-1 rounded-full">
                  {product.category_name}
                </span>
                {product.location && (
                  <span className="flex items-center">
                    📍 {product.location}
                  </span>
                )}
              </div>

              <div className="text-4xl font-bold text-primary-600 mb-6">
                {parseFloat(product.price).toLocaleString('mk-MK')} ден.
              </div>
            </div>

            <div>
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Опис</h3>
              <p className="text-gray-700 leading-relaxed whitespace-pre-line">
                {product.description}
              </p>
            </div>

            {product.stock_quantity !== undefined && (
              <div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Достапност</h3>
                <span className={`px-3 py-1 rounded-full text-sm ${
                  product.stock_quantity > 0 
                    ? 'bg-green-100 text-green-800' 
                    : 'bg-red-100 text-red-800'
                }`}>
                  {product.stock_quantity > 0 
                    ? `${product.stock_quantity} парчиња достапни`
                    : 'Нема на залиха'
                  }
                </span>
              </div>
            )}

            {/* Seller Info */}
            <div className="border-t pt-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Продавач</h3>
              <div className="bg-gray-50 p-4 rounded-lg">
                <div className="flex items-center space-x-3 mb-3">
                  <div className="w-12 h-12 bg-primary-600 rounded-full flex items-center justify-center text-white font-bold">
                    {product.first_name?.charAt(0)}{product.last_name?.charAt(0)}
                  </div>
                  <div>
                    <p className="font-semibold text-gray-900">
                      {product.first_name} {product.last_name}
                    </p>
                    <p className="text-gray-600">@{product.seller_name}</p>
                  </div>
                </div>
                
                {product.phone && (
                  <p className="text-sm text-gray-600 mb-2">
                    📞 {product.phone}
                  </p>
                )}
                
                {product.email && (
                  <p className="text-sm text-gray-600">
                    ✉️ {product.email}
                  </p>
                )}
              </div>
            </div>

            {/* Contact Buttons */}
            <div className="space-y-3">
              <button className="w-full bg-primary-600 hover:bg-primary-700 text-white font-bold py-3 px-6 rounded-lg transition-colors">
                Контактирај го продавачот
              </button>
              
              <button className="w-full bg-green-600 hover:bg-green-700 text-white font-bold py-3 px-6 rounded-lg transition-colors">
                Пораќај преку WhatsApp
              </button>
            </div>
          </div>
        </div>
      </div>

      {/* Product Details */}
      <div className="mt-8 bg-white rounded-lg shadow-lg p-6">
        <h3 className="text-xl font-bold text-gray-900 mb-4">Детали за производот</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
          <div>
            <span className="font-medium text-gray-900">Категорија:</span>
            <span className="ml-2 text-gray-700">{product.category_name}</span>
          </div>
          <div>
            <span className="font-medium text-gray-900">Продавач:</span>
            <span className="ml-2 text-gray-700">{product.seller_name}</span>
          </div>
          {product.location && (
            <div>
              <span className="font-medium text-gray-900">Локација:</span>
              <span className="ml-2 text-gray-700">{product.location}</span>
            </div>
          )}
          <div>
            <span className="font-medium text-gray-900">Објавен:</span>
            <span className="ml-2 text-gray-700">
              {new Date(product.created_at).toLocaleDateString('mk-MK')}
            </span>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ProductDetail;
