import React, { useState, useEffect, useCallback } from 'react';
import { useParams } from 'react-router-dom';
import { useProducts } from '../context/ProductContext';
import { useAuth } from '../context/AuthContext';
import { useCart } from '../context/CartContext';
import ReviewForm from '../components/ReviewForm';
import ReviewList from '../components/ReviewList';
import StarRating from '../components/StarRating';
import api from '../services/api';

const ProductDetail = () => {
  const { id } = useParams();
  const { fetchProduct } = useProducts();
  const { isAuthenticated } = useAuth();
  const { addToCart } = useCart();
  const [product, setProduct] = useState(null);
  const [loading, setLoading] = useState(true);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);
  const [showShareMessage, setShowShareMessage] = useState(false);
  const [quantity, setQuantity] = useState(1);
  const [isAddingToCart, setIsAddingToCart] = useState(false);
  const [reviews, setReviews] = useState([]);
  const [reviewsLoading, setReviewsLoading] = useState(false);
  const [showReviewForm, setShowReviewForm] = useState(false);
  const [averageRating, setAverageRating] = useState(0);
  const [totalReviews, setTotalReviews] = useState(0);

  const copyProductLink = () => {
    const productUrl = window.location.href;
    navigator.clipboard.writeText(productUrl).then(() => {
      setShowShareMessage(true);
      setTimeout(() => setShowShareMessage(false), 3000);
    }).catch(() => {
      // Fallback for older browsers
      const textArea = document.createElement('textarea');
      textArea.value = productUrl;
      document.body.appendChild(textArea);
      textArea.select();
      document.execCommand('copy');
      document.body.removeChild(textArea);
      setShowShareMessage(true);
      setTimeout(() => setShowShareMessage(false), 3000);
    });
  };

  const handleAddToCart = async () => {
    if (!isAuthenticated()) {
      alert('Please log in to add items to cart');
      return;
    }
    
    if (!product || product.stock_quantity < quantity) {
      alert('Not enough stock available');
      return;
    }
    
    setIsAddingToCart(true);
    const result = await addToCart(product.id, quantity);
    
    if (result.success) {
      alert(result.message || `${quantity} item(s) added to cart!`);
    } else {
      alert(result.message || 'Failed to add product to cart');
    }
    
    setIsAddingToCart(false);
  };

  const handleQuantityChange = (newQuantity) => {
    if (newQuantity >= 1 && newQuantity <= (product?.stock_quantity || 0)) {
      setQuantity(newQuantity);
    }
  };

  const fetchReviews = useCallback(async () => {
    if (!id) return;
    
    setReviewsLoading(true);
    try {
      const response = await api.get(`/reviews/product/${id}`);
      if (response.data.success) {
        setReviews(response.data.reviews);
        setAverageRating(response.data.averageRating || 0);
        setTotalReviews(response.data.totalReviews || 0);
      }
    } catch (error) {
      console.error('Error fetching reviews:', error);
    } finally {
      setReviewsLoading(false);
    }
  }, [id]);

  const handleReviewAdded = (newReview) => {
    setReviews(prev => [newReview, ...prev]);
    setShowReviewForm(false);   // ja zatvara formata
    setTotalReviews(prev => prev + 1);
    // Recalculate average rating
    const newAverage = (averageRating * totalReviews + newReview.rating) / (totalReviews + 1);
    setAverageRating(newAverage);
  };

  const handleReviewUpdate = (updatedReview) => {
    setReviews(prev => prev.map(review => 
      review.id === updatedReview.id ? updatedReview : review
    ));
    // You might want to recalculate average rating here too
    fetchReviews(); // Simple approach - refetch all reviews
  };

  const handleReviewDelete = (reviewId) => {
    setReviews(prev => prev.filter(review => review.id !== reviewId));
    setTotalReviews(prev => prev - 1);
    // Recalculate average rating or refetch
    fetchReviews();
  };

  useEffect(() => {
    const loadProduct = async () => {
      const result = await fetchProduct(id);
      if (result.success) {
        setProduct(result.data);
      }
      setLoading(false);
    };

    loadProduct();
    fetchReviews();
  }, [id, fetchProduct, fetchReviews]);

  if (loading) {
    return (
      <div className="flex justify-center items-center min-h-screen">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-primary-600"></div>
      </div>
    );
  }

  if (!product) {
    return (
      <div className="text-center py-12">
        <h2 className="text-2xl font-bold text-gray-900 mb-4">Производот не е пронајден</h2>
        <p className="text-gray-600">Производот што го барате не постои или е отстранет.</p>
      </div>
    );
  }

  const images = product.images && product.images.length > 0 
    ? product.images.map(img => {
        if (img.startsWith('http')) return img;
        return img.startsWith('/') ? `http://localhost:5001${img}` : `http://localhost:5001/uploads/products/${img}`;
      })
    : ['/placeholder-image.svg'];

  return (
    <div className="max-w-6xl mx-auto">
      <div className="bg-white rounded-lg shadow-lg overflow-hidden">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 p-6">
          {/* Image Gallery */}
          <div className="space-y-4">
            <div className="aspect-w-16 aspect-h-12 bg-gray-200 rounded-lg overflow-hidden">
              <img
                src={images[currentImageIndex]}
                alt={product.title}
                className="w-full h-96 object-cover"
                onError={(e) => {
                  e.target.src = '/placeholder-image.jpg';
                }}
              />
            </div>
            
            {images.length > 1 && (
              <div className="grid grid-cols-4 gap-2">
                {images.map((image, index) => (
                  <button
                    key={index}
                    onClick={() => setCurrentImageIndex(index)}
                    className={`aspect-w-1 aspect-h-1 rounded-lg overflow-hidden border-2 ${
                      currentImageIndex === index ? 'border-primary-600' : 'border-transparent'
                    }`}
                  >
                    <img
                      src={image}
                      alt={`${product.title} ${index + 1}`}
                      className="w-full h-20 object-cover"
                    />
                  </button>
                ))}
              </div>
            )}
          </div>

          {/* Product Info */}
          <div className="space-y-6">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">
                {product.title}
              </h1>
              
              <div className="flex items-center space-x-4 text-sm text-gray-600 mb-4">
                <span className="bg-primary-100 text-primary-800 px-3 py-1 rounded-full">
                  {product.category_name}
                </span>
                {product.location && (
                  <span className="flex items-center">
                    📍 {product.location}
                  </span>
                )}
              </div>

              <div className="text-4xl font-bold text-primary-600 mb-6">
                {parseFloat(product.price).toLocaleString('mk-MK')} ден.
              </div>
            </div>

            <div>
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Опис</h3>
              <p className="text-gray-700 leading-relaxed whitespace-pre-line">
                {product.description}
              </p>
            </div>

            {product.stock_quantity !== undefined && (
              <div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Достапност</h3>
                <span className={`px-3 py-1 rounded-full text-sm ${
                  product.stock_quantity > 0 
                    ? 'bg-green-100 text-green-800' 
                    : 'bg-red-100 text-red-800'
                }`}>
                  {product.stock_quantity > 0 
                    ? `${product.stock_quantity} парчиња достапни`
                    : 'Нема на залиха'
                  }
                </span>
              </div>
            )}

            {/* Seller Info */}
            <div className="border-t pt-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-3">Продавач</h3>
              <div className="bg-gray-50 p-4 rounded-lg">
                <div className="flex items-center space-x-3 mb-3">
                  <div className="w-12 h-12 bg-primary-600 rounded-full flex items-center justify-center text-white font-bold">
                    {product.first_name?.charAt(0)}{product.last_name?.charAt(0)}
                  </div>
                  <div>
                    <p className="font-semibold text-gray-900">
                      {product.first_name} {product.last_name}
                    </p>
                    <p className="text-gray-600">@{product.seller_name}</p>
                  </div>
                </div>
                
                {product.phone && (
                  <p className="text-sm text-gray-600 mb-2">
                    📞 {product.phone}
                  </p>
                )}
                
                {product.email && (
                  <p className="text-sm text-gray-600">
                    ✉️ {product.email}
                  </p>
                )}
              </div>
            </div>

            {/* Add to Cart Section */}
            {isAuthenticated() && (
              <div className="bg-gray-50 p-4 rounded-lg space-y-4">
                <h3 className="text-lg font-semibold text-gray-900">
                  Додај во кошничка
                </h3>
                
                {/* Stock Information */}
                <div className="text-sm">
                  {product.stock_quantity > 0 ? (
                    <span className="text-green-600 font-medium">
                      ✓ {product.stock_quantity} на залиха
                    </span>
                  ) : (
                    <span className="text-red-600 font-medium">✗ Нема на залиха</span>
                  )}
                </div>
                
                {/* Quantity Selector */}
                {product.stock_quantity > 0 && (
                  <div className="flex items-center space-x-4">
                    <label className="text-sm font-medium text-gray-700">
                      Количина:
                    </label>
                    <div className="flex items-center border border-gray-300 rounded-md">
                      <button
                        onClick={() => handleQuantityChange(quantity - 1)}
                        disabled={quantity <= 1}
                        className="px-3 py-2 hover:bg-gray-100 disabled:opacity-50 disabled:cursor-not-allowed"
                      >
                        -
                      </button>
                      <span className="px-4 py-2 border-l border-r border-gray-300 min-w-[50px] text-center">
                        {quantity}
                      </span>
                      <button
                        onClick={() => handleQuantityChange(quantity + 1)}
                        disabled={quantity >= product.stock_quantity}
                        className="px-3 py-2 hover:bg-gray-100 disabled:opacity-50 disabled:cursor-not-allowed"
                      >
                        +
                      </button>
                    </div>
                    <span className="text-sm text-gray-500">
                      (макс {product.stock_quantity})
                    </span>
                  </div>
                )}
                
                {/* Add to Cart Button */}
                <button
                  onClick={handleAddToCart}
                  disabled={isAddingToCart || product.stock_quantity < 1 || quantity < 1}
                  className={`w-full py-3 px-6 rounded-lg font-bold transition-colors ${
                    product.stock_quantity < 1
                      ? 'bg-gray-300 text-gray-500 cursor-not-allowed'
                      : isAddingToCart
                      ? 'bg-blue-400 text-white cursor-not-allowed'
                      : 'bg-blue-600 hover:bg-blue-700 text-white'
                  }`}
                >
                  {isAddingToCart ? (
                    <div className="flex items-center justify-center space-x-2">
                      <div className="w-4 h-4 animate-spin border-2 border-white border-t-transparent rounded-full"></div>
                      <span>Се додава...</span>
                    </div>
                  ) : product.stock_quantity < 1 ? (
                    'Нема на залиха'
                  ) : (
                    `🛒 Додај ${quantity} во кошничка`
                  )}
                </button>
                
                {/* Total Price */}
                {product.stock_quantity > 0 && (
                  <div className="text-center text-lg font-semibold text-gray-900">
                    Вкупно: {(parseFloat(product.price) * quantity).toLocaleString('mk-MK')} ден.
                  </div>
                )}
              </div>
            )}

            {/* Contact Buttons */}
            <div className="space-y-3">
              <button className="w-full bg-primary-600 hover:bg-primary-700 text-white font-bold py-3 px-6 rounded-lg transition-colors">
                Контактирај го продавачот
              </button>
              
              <button className="w-full bg-green-600 hover:bg-green-700 text-white font-bold py-3 px-6 rounded-lg transition-colors">
                Пораќај преку WhatsApp
              </button>
              
              <button 
                onClick={copyProductLink}
                className="w-full bg-blue-600 hover:bg-blue-700 text-white font-bold py-3 px-6 rounded-lg transition-colors flex items-center justify-center gap-2"
              >
                📋 Сподели линк
              </button>
              
              {showShareMessage && (
                <div className="text-center p-3 bg-green-100 text-green-800 rounded-lg">
                  ✅ Линкот е копиран!
                </div>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Product Details */}
      <div className="mt-8 bg-white rounded-lg shadow-lg p-6">
        <h3 className="text-xl font-bold text-gray-900 mb-4">Детали за производот</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
          <div>
            <span className="font-medium text-gray-900">Категорија:</span>
            <span className="ml-2 text-gray-700">{product.category_name}</span>
          </div>
          <div>
            <span className="font-medium text-gray-900">Продавач:</span>
            <span className="ml-2 text-gray-700">{product.seller_name}</span>
          </div>
          {product.location && (
            <div>
              <span className="font-medium text-gray-900">Локација:</span>
              <span className="ml-2 text-gray-700">{product.location}</span>
            </div>
          )}
          <div>
            <span className="font-medium text-gray-900">Објавен:</span>
            <span className="ml-2 text-gray-700">
              {new Date(product.created_at).toLocaleDateString('mk-MK')}
            </span>
          </div>
        </div>
      </div>

      {/* Reviews Section */}
      <div className="mt-8 bg-white rounded-lg shadow-lg p-6">
        <div className="flex items-center justify-between mb-6">
          <div>
            <h3 className="text-xl font-bold text-gray-900">Customer Reviews</h3>
            <div className="flex items-center space-x-4 mt-2">
              <div className="flex items-center space-x-2">
                <StarRating rating={averageRating} readonly={true} showRatingText={false} />
                <span className="text-sm text-gray-600">
                  {averageRating > 0 ? averageRating.toFixed(1) : '0'} out of 5
                </span>
              </div>
              <span className="text-sm text-gray-500">
                ({totalReviews} {totalReviews === 1 ? 'review' : 'reviews'})
              </span>
            </div>
          </div>
          
          {isAuthenticated() && (
            <button
              onClick={() => setShowReviewForm(!showReviewForm)}
              className="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700"
            >
              {showReviewForm ? 'Cancel' : 'Write Review'}
            </button>
          )}
        </div>

        {/* Review Form */}
        {showReviewForm && (
          <div className="mb-6">
            <ReviewForm
              productId={product.id}
              onReviewAdded={handleReviewAdded}
              onCancel={() => setShowReviewForm(false)}
            />
          </div>
        )}

        {/* Reviews List */}
        {reviewsLoading ? (
          <div className="text-center py-8">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
            <p className="mt-2 text-gray-600">Loading reviews...</p>
          </div>
        ) : (
          <ReviewList
            reviews={reviews}
            onReviewUpdate={handleReviewUpdate}
            onReviewDelete={handleReviewDelete}
          />
        )}
      </div>
    </div>
  );
};

export default ProductDetail;
