const passport = require('passport');
const GoogleStrategy = require('passport-google-oauth20').Strategy;
const { pool } = require('./database');

passport.use(
  new GoogleStrategy(
    {
      clientID: process.env.GOOGLE_CLIENT_ID,
      clientSecret: process.env.GOOGLE_CLIENT_SECRET,
      callbackURL: process.env.GOOGLE_CALLBACK_URL
    },
    async (accessToken, refreshToken, profile, done) => {
      try {
        // Check if user exists with this Google ID
        let user = await pool.query(
          'SELECT * FROM users WHERE google_id = $1',
          [profile.id]
        );

        if (user.rows.length > 0) {
          // User exists, return user
          return done(null, user.rows[0]);
        }

        // Check if user exists with this email
        user = await pool.query(
          'SELECT * FROM users WHERE email = $1',
          [profile.emails[0].value]
        );

        if (user.rows.length > 0) {
          // User with this email exists
          // Check if they have a password (traditional account)
          if (user.rows[0].password) {
            // Account was created with password, don't allow OAuth login
            return done(null, false, { 
              message: 'An account with this email already exists. Please login with your password instead.' 
            });
          }
          
          // User exists but has no password (OAuth-only account), link Google account
          const updatedUser = await pool.query(
            'UPDATE users SET google_id = $1 WHERE id = $2 RETURNING *',
            [profile.id, user.rows[0].id]
          );
          return done(null, updatedUser.rows[0]);
        }

        // Create new user
        const newUser = await pool.query(
          `INSERT INTO users (
            google_id, 
            username, 
            email, 
            first_name, 
            last_name, 
            is_seller
          ) VALUES ($1, $2, $3, $4, $5, $6) RETURNING *`,
          [
            profile.id,
            profile.emails[0].value.split('@')[0] + '_' + Date.now(), // Generate unique username
            profile.emails[0].value,
            profile.name.givenName,
            profile.name.familyName,
            false
          ]
        );

        return done(null, newUser.rows[0]);
      } catch (error) {
        console.error('Error in Google Strategy:', error);
        return done(error, null);
      }
    }
  )
);

passport.serializeUser((user, done) => {
  done(null, user.id);
});

passport.deserializeUser(async (id, done) => {
  try {
    const user = await pool.query('SELECT * FROM users WHERE id = $1', [id]);
    done(null, user.rows[0]);
  } catch (error) {
    done(error, null);
  }
});

module.exports = passport;
