// Seller Dashboard Page
// - Purpose: provide a seller-only view of orders that include products from the authenticated seller
// - Features:
//   1) List orders filtered to this seller's products
//   2) Show customer information and quick order summary
//   3) Expand to view order items, shipping address, and notes
//   4) Allow status updates (confirm, process, ship, deliver, cancel)

import React, { useState, useEffect, useCallback } from 'react';
import api from '../services/api';

const SellerDashboard = () => {
    // ----- Component state -----
    // orders: list of orders returned from backend (already filtered to seller)
    // loading / error: UI state for network requests
    // filter: UI filter for order status
    // expandedOrder: which order row is expanded to show details
    const [orders, setOrders] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [filter, setFilter] = useState('all'); // all, pending, confirmed, processing, shipped, delivered, cancelled
    const [expandedOrder, setExpandedOrder] = useState(null);

    // ----- Fetch orders -----
    // fetchAllOrders: calls `/orders/admin/all` and applies `filter` on the client side
    const fetchAllOrders = useCallback(async () => {
        try {
            setLoading(true);
            const response = await api.get('/orders/admin/all');
            if (response.data.success) {
                let filteredOrders = response.data.orders;
                if (filter !== 'all') {
                    filteredOrders = response.data.orders.filter(order => order.status === filter);
                }
                setOrders(filteredOrders);
            } else {
                setError('Failed to fetch orders');
            }
        } catch (error) {
            console.error('Error fetching orders:', error);
            setError('Failed to fetch orders');
        } finally {
            setLoading(false);
        }
    }, [filter]);

    useEffect(() => {
        fetchAllOrders();
    }, [fetchAllOrders]);

    // ----- Update order status -----
    // handleUpdateOrderStatus: sends updated status to backend and refreshes order list
    const handleUpdateOrderStatus = async (orderId, newStatus) => {
        try {
            const response = await api.put(`/orders/admin/${orderId}/status`, {
                status: newStatus
            });

            if (response.data.success) {
                fetchAllOrders(); // Refresh the list
                alert(`Order #${orderId} status updated to ${newStatus}`);
            } else {
                alert(response.data.message || 'Failed to update order status');
            }
        } catch (error) {
            console.error('Error updating order status:', error);
            alert('Failed to update order status');
        }
    };

    // ----- UI helpers -----
    // getStatusColor: returns CSS classes for order status badge
    const getStatusColor = (status) => {
        switch (status.toLowerCase()) {
            case 'pending':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'confirmed':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'processing':
                return 'bg-purple-100 text-purple-800 border-purple-200';
            case 'shipped':
                return 'bg-indigo-100 text-indigo-800 border-indigo-200';
            case 'delivered':
                return 'bg-green-100 text-green-800 border-green-200';
            case 'cancelled':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    // formatDate: formats date string to local date string
    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    // getAvailableStatuses: returns next possible statuses for an order based on its current status
    const getAvailableStatuses = (currentStatus) => {
        const statusFlow = {
            'pending': ['confirmed', 'cancelled'],
            'confirmed': ['processing', 'cancelled'],
            'processing': ['shipped'],
            'shipped': ['delivered'],
            'delivered': [],
            'cancelled': []
        };
        return statusFlow[currentStatus] || [];
    };

    // ----- Render -----
    if (loading) {
        return (
            <div className="min-h-screen bg-gray-50 flex items-center justify-center">
                <div className="text-center">
                    <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                    <p className="mt-4 text-gray-600">Loading orders...</p>
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="min-h-screen bg-gray-50 flex items-center justify-center">
                <div className="text-center">
                    <p className="text-red-600 mb-4">{error}</p>
                    <button
                        onClick={fetchAllOrders}
                        className="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700"
                    >
                        Try Again
                    </button>
                </div>
            </div>
        );
    }

    return (
        <div className="min-h-screen bg-gray-50 py-8">
            <div className="max-w-7xl mx-auto px-4">
                <div className="mb-8">
                    <h1 className="text-3xl font-bold text-gray-800">Seller Dashboard</h1>
                    <p className="text-gray-600 mt-2">Manage and track all customer orders</p>
                </div>

                {/* Filter Tabs */}
                <div className="bg-white rounded-lg shadow-md mb-6">
                    <div className="border-b border-gray-200">
                        <nav className="flex space-x-8 px-6">
                            {['all', 'pending', 'confirmed', 'processing', 'shipped', 'delivered', 'cancelled'].map((status) => (
                                <button
                                    key={status}
                                    onClick={() => setFilter(status)}
                                    className={`py-4 px-1 border-b-2 font-medium text-sm capitalize ${
                                        filter === status
                                            ? 'border-blue-500 text-blue-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    {status} ({orders.filter(order => status === 'all' || order.status === status).length})
                                </button>
                            ))}
                        </nav>
                    </div>
                </div>

                {/* Orders Table */}
                <div className="bg-white rounded-lg shadow-md overflow-hidden">
                    <div className="overflow-x-auto">
                        <table className="min-w-full divide-y divide-gray-200">
                            <thead className="bg-gray-50">
                                <tr>
                                    {/* Table header: basic order meta */}
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Order ID
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Customer
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Date
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Total
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Status
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Actions
                                    </th>
                                </tr>
                            </thead>
                            <tbody className="bg-white divide-y divide-gray-200">
                                {orders.map((order) => (
                                    <React.Fragment key={order.id}>
                                        {/* Order row: summary information */}
                                        <tr className="hover:bg-gray-50">
                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                                #{order.id}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                {/* Customer name, phone and email (from joined users table) */}
                                                {order.first_name && order.last_name 
                                                    ? `${order.first_name} ${order.last_name}` 
                                                    : order.shipping_address?.fullName || 'N/A'}
                                                <br />
                                                <span className="text-xs text-gray-500">{order.phone}</span>
                                                <br />
                                                <span className="text-xs text-gray-400">{order.email}</span>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                                {formatDate(order.created_at)}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                {parseFloat(order.seller_total ?? order.total_amount ?? 0).toLocaleString('mk-MK')} ден.
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <span className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium border ${getStatusColor(order.status)}`}>
                                                    {order.status.charAt(0).toUpperCase() + order.status.slice(1)}
                                                </span>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                                                {/* Actions: seller can advance order status according to business rules */}
                                                {getAvailableStatuses(order.status).map((nextStatus) => (
                                                    <button
                                                        key={nextStatus}
                                                        onClick={() => handleUpdateOrderStatus(order.id, nextStatus)}
                                                        className={`inline-flex items-center px-3 py-1 border border-transparent text-sm leading-5 font-medium rounded-md text-white ${
                                                            nextStatus === 'cancelled' 
                                                                ? 'bg-red-600 hover:bg-red-700' 
                                                                : 'bg-blue-600 hover:bg-blue-700'
                                                        } focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500`}
                                                    >
                                                        Mark as {nextStatus}
                                                    </button>
                                                ))}
                                                
                                                {/* Toggle expanded details inline instead of using alert() */}
                                                <button
                                                    onClick={() => setExpandedOrder(expandedOrder === order.id ? null : order.id)}
                                                    className="inline-flex items-center px-3 py-1 border border-gray-300 text-sm leading-5 font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                                >
                                                    {expandedOrder === order.id ? 'Hide Details' : 'View Details'}
                                                </button>
                                            </td>
                                        </tr>
                                        
                                        {/* Expanded Order Details Row: displayed inline when the seller clicks View Details */}
                                        {expandedOrder === order.id && (
                                            <tr>
                                                <td colSpan="6" className="px-6 py-4 bg-gray-50">
                                                    <div className="space-y-4">
                                                        {/* Order Items */}
                                                        <div>
                                                            <h4 className="font-semibold text-gray-800 mb-3">Order Items</h4>
                                                            <div className="space-y-2">
                                                                {order.order_items && order.order_items.map((item, index) => (
                                                                    <div key={index} className="flex items-center space-x-4 bg-white p-3 rounded-md">
                                                                        {item.image_url && (
                                                                            <img 
                                                                                src={`http://localhost:5001${item.image_url}`}
                                                                                alt={item.product_title}
                                                                                className="w-12 h-12 object-cover rounded-md"
                                                                            />
                                                                        )}
                                                                        <div className="flex-1">
                                                                            <p className="font-medium text-gray-800">{item.product_title}</p>
                                                                            <p className="text-sm text-gray-600">
                                                                                Quantity: {item.quantity} × {parseFloat(item.price_at_time || 0).toLocaleString('mk-MK')} ден.
                                                                            </p>
                                                                        </div>
                                                                        <div className="text-right">
                                                                            <p className="font-medium text-gray-800">
                                                                                {(parseFloat(item.price_at_time || 0) * item.quantity).toLocaleString('mk-MK')} ден.
                                                                            </p>
                                                                        </div>
                                                                    </div>
                                                                ))}
                                                            </div>
                                                        </div>
                                                        
                                                        {/* Shipping Information */}
                                                        {order.shipping_address && (
                                                            <div>
                                                                <h4 className="font-semibold text-gray-800 mb-2">Shipping Address</h4>
                                                                <div className="bg-white p-3 rounded-md text-sm text-gray-700">
                                                                    <p>{order.shipping_address?.fullName}</p>
                                                                    <p>{order.shipping_address?.street}</p>
                                                                    <p>{order.shipping_address ? `${order.shipping_address.city || ''}${order.shipping_address.postalCode ? ', ' + order.shipping_address.postalCode : ''}` : ''}</p>
                                                                    <p>{order.shipping_address?.country}</p>
                                                                </div>
                                                            </div>
                                                        )}
                                                        
                                                        {/* Order Notes */}
                                                        {order.notes && (
                                                            <div>
                                                                <h4 className="font-semibold text-gray-800 mb-2">Order Notes</h4>
                                                                <div className="bg-white p-3 rounded-md text-sm text-gray-700">
                                                                    {order.notes}
                                                                </div>
                                                            </div>
                                                        )}
                                                    </div>
                                                </td>
                                            </tr>
                                        )}
                                    </React.Fragment>
                                ))}
                            </tbody>
                        </table>
                    </div>

                    {orders.length === 0 && (
                        <div className="text-center py-12">
                            <div className="mb-4">
                                <svg className="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
                                </svg>
                            </div>
                            <h3 className="text-lg font-medium text-gray-900 mb-2">No orders found</h3>
                            <p className="text-gray-600">
                                {filter === 'all' ? 'No orders have been placed yet.' : `No ${filter} orders found.`}
                            </p>
                        </div>
                    )}
                </div>
            </div>
        </div>
    );
};

export default SellerDashboard;