#!/bin/bash

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

echo -e "${GREEN}Starting Kupi.mk Development Environment...${NC}"
echo ""

# Function to check if command was successful
check_status() {
    if [ $? -ne 0 ]; then
        echo -e "${RED}$1${NC}"
        exit 1
    fi
}

# Install backend dependencies
echo -e "${YELLOW}Installing Backend Dependencies...${NC}"
cd backend
npm install
check_status "Failed to install backend dependencies"

# Install frontend dependencies
echo -e "${YELLOW}Installing Frontend Dependencies...${NC}"
cd ../frontend
npm install
check_status "Failed to install frontend dependencies"

# Start backend server in background
echo -e "${YELLOW}Starting Backend Server...${NC}"
cd ../backend
npm run dev > ../backend.log 2>&1 &
BACKEND_PID=$!
echo "Backend PID: $BACKEND_PID"

# Wait for backend to start
echo -e "${YELLOW}Waiting for backend to start...${NC}"
sleep 3

# Check if backend is still running
if ! kill -0 $BACKEND_PID 2>/dev/null; then
    echo -e "${RED}Backend failed to start. Check backend.log for details.${NC}"
    exit 1
fi

# Start frontend server
echo -e "${YELLOW}Starting Frontend Server...${NC}"
cd ../frontend
export HOST=0.0.0.0
npm start > ../frontend.log 2>&1 &
FRONTEND_PID=$!
echo "Frontend PID: $FRONTEND_PID"

echo ""
echo -e "${GREEN}Both servers are starting up...${NC}"
echo -e "${CYAN}Backend: http://localhost:5000${NC}"
echo -e "${CYAN}Frontend: http://localhost:3000${NC}"
echo -e "${CYAN}Network: http://0.0.0.0:3000${NC}"
echo ""
echo -e "${GREEN}Process IDs:${NC}"
echo "Backend PID: $BACKEND_PID"
echo "Frontend PID: $FRONTEND_PID"
echo ""
echo -e "${YELLOW}To stop the servers, run:${NC}"
echo "kill $BACKEND_PID $FRONTEND_PID"
echo ""
echo -e "${YELLOW}Logs are available in:${NC}"
echo "Backend: $(pwd)/../backend.log"
echo "Frontend: $(pwd)/../frontend.log"
echo ""
echo -e "${GREEN}Press Ctrl+C to stop monitoring or close this terminal.${NC}"

# Monitor processes
while true; do
    if ! kill -0 $BACKEND_PID 2>/dev/null; then
        echo -e "${RED}Backend process died!${NC}"
        break
    fi
    if ! kill -0 $FRONTEND_PID 2>/dev/null; then
        echo -e "${RED}Frontend process died!${NC}"
        break
    fi
    sleep 5
done