import React, { useState } from 'react';
import { useCart } from '../context/CartContext';

const CartItem = ({ item }) => {
  const { updateCartItem, removeFromCart } = useCart();
  const [isUpdating, setIsUpdating] = useState(false);
  const [isRemoving, setIsRemoving] = useState(false);

  const getImageUrl = (imagePath) => {
    if (!imagePath) return '/placeholder-image.svg';
    if (imagePath.startsWith('http')) return imagePath;
    if (imagePath.startsWith('/')) {
      return `http://localhost:5001${imagePath}`;
    }
    return `http://localhost:5001/uploads/products/${imagePath}`;
  };

  const imageUrl = item.images && item.images.length > 0 
    ? getImageUrl(item.images[0])
    : '/placeholder-image.svg';

  const handleQuantityChange = async (newQuantity) => {
    if (newQuantity < 1) return;
    
    setIsUpdating(true);
    const result = await updateCartItem(item.id, newQuantity);
    if (!result.success) {
      alert(result.message || 'Failed to update quantity');
    }
    setIsUpdating(false);
  };

  const handleRemove = async () => {
    if (window.confirm('Remove this item from your cart?')) {
      setIsRemoving(true);
      const result = await removeFromCart(item.id);
      if (!result.success) {
        alert(result.message || 'Failed to remove item');
        setIsRemoving(false);
      }
    }
  };

  const itemTotal = parseFloat(item.price) * item.quantity;

  return (
    <div className={`flex items-center space-x-4 py-4 border-b border-gray-200 ${isRemoving ? 'opacity-50' : ''}`}>
      {/* Product Image */}
      <div className="flex-shrink-0 w-16 h-16 bg-gray-200 rounded-md overflow-hidden">
        <img
          src={imageUrl}
          alt={item.title}
          className="w-full h-full object-cover"
          onError={(e) => {
            e.target.src = '/placeholder-image.svg';
          }}
        />
      </div>

      {/* Product Details */}
      <div className="flex-1 min-w-0">
        <h4 className="text-sm font-medium text-gray-900 truncate">
          {item.title}
        </h4>
        <p className="text-sm text-gray-500">
          {item.category_name}
        </p>
        <p className="text-sm font-medium text-gray-900">
          {parseFloat(item.price).toLocaleString()} МКД
        </p>
      </div>

      {/* Quantity Controls */}
      <div className="flex items-center space-x-2">
        <button
          onClick={() => handleQuantityChange(item.quantity - 1)}
          disabled={isUpdating || item.quantity <= 1}
          className="w-8 h-8 rounded-full border border-gray-300 flex items-center justify-center hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          -
        </button>
        
        <span className="w-8 text-center text-sm font-medium">
          {isUpdating ? '...' : item.quantity}
        </span>
        
        <button
          onClick={() => handleQuantityChange(item.quantity + 1)}
          disabled={isUpdating || item.quantity >= item.stock_quantity}
          className="w-8 h-8 rounded-full border border-gray-300 flex items-center justify-center hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          +
        </button>
      </div>

      {/* Item Total */}
      <div className="text-right">
        <p className="text-sm font-medium text-gray-900">
          {itemTotal.toLocaleString()} МКД
        </p>
        {item.quantity > 1 && (
          <p className="text-xs text-gray-500">
            {parseFloat(item.price).toLocaleString()} × {item.quantity}
          </p>
        )}
      </div>

      {/* Remove Button */}
      <button
        onClick={handleRemove}
        disabled={isRemoving}
        className="flex-shrink-0 p-2 text-gray-400 hover:text-red-500 transition-colors disabled:opacity-50"
        aria-label="Remove item"
      >
        {isRemoving ? (
          <div className="w-4 h-4 animate-spin border-2 border-gray-400 border-t-transparent rounded-full"></div>
        ) : (
          <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
          </svg>
        )}
      </button>
    </div>
  );
};

export default CartItem;