import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { useAuth } from '../context/AuthContext';
import { useCart } from '../context/CartContext';
import StarRating from './StarRating';

const ProductCard = ({ product }) => {
  const { isAuthenticated } = useAuth();
  const { addToCart } = useCart();
  const [isAddingToCart, setIsAddingToCart] = useState(false);
  
  const getImageUrl = (imagePath) => {
    if (!imagePath) return '/placeholder-image.svg';
    
    // If it's already a full URL, return as is
    if (imagePath.startsWith('http')) return imagePath;
    
    // If it starts with /, add the backend URL
    if (imagePath.startsWith('/')) {
      return `http://localhost:5001${imagePath}`;
    }
    
    // Otherwise assume it's a relative path
    return `http://localhost:5001/uploads/products/${imagePath}`;
  };

  const imageUrl = product.images && product.images.length > 0 
    ? getImageUrl(product.images[0])
    : '/placeholder-image.svg';

  console.log('Product images:', product.images); // Debug log
  console.log('Image URL:', imageUrl); // Debug log

  const handleAddToCart = async (e) => {
    e.preventDefault();
    e.stopPropagation();
    
    if (!isAuthenticated()) {
      alert('Please log in to add items to cart');
      return;
    }
    
    if (product.stock_quantity < 1) {
      alert('This product is out of stock');
      return;
    }
    
    setIsAddingToCart(true);
    const result = await addToCart(product.id, 1);
    
    if (result.success) {
      // Could show a toast notification here instead
      alert(result.message || 'Product added to cart!');
    } else {
      alert(result.message || 'Failed to add product to cart');
    }
    
    setIsAddingToCart(false);
  };

  return (
    <div className="bg-white rounded-lg shadow-md overflow-hidden hover:shadow-lg transition-shadow duration-300 group">
      {/* Product Image and Info - Clickable Link */}
      <Link to={`/product/${product.id}`} className="block">
        <div className="aspect-w-16 aspect-h-9 bg-gray-200">
          <img
            src={imageUrl}
            alt={product.title}
            className="w-full h-48 object-cover group-hover:scale-105 transition-transform duration-300"
            onError={(e) => {
              console.error('Image failed to load:', e.target.src);
              e.target.src = '/placeholder-image.svg';
            }}
            onLoad={() => {
              console.log('Image loaded successfully:', imageUrl);
            }}
          />
        </div>
        
        <div className="p-4 pb-2">
          <h3 className="text-lg font-semibold text-gray-900 mb-2 line-clamp-2">
            {product.title}
          </h3>
          
          <p className="text-gray-600 text-sm mb-3 line-clamp-2">
            {product.description}
          </p>
          
          <div className="flex justify-between items-center">
            <span className="text-2xl font-bold text-primary-600">
              {parseFloat(product.price).toLocaleString('mk-MK')} ден.
            </span>
            
            {product.location && (
              <span className="text-xs text-gray-500 bg-gray-100 px-2 py-1 rounded">
                📍 {product.location}
              </span>
            )}
          </div>
          
          {/* Rating Display */}
          {product.average_rating && product.average_rating > 0 && (
            <div className="mt-2">
              <StarRating 
                rating={product.average_rating} 
                readonly={true} 
                size="sm"
                showRatingText={false}
              />
              <span className="text-xs text-gray-500 ml-2">
                ({product.review_count || 0} reviews)
              </span>
            </div>
          )}
          
          <div className="mt-3 flex justify-between items-center text-sm text-gray-500">
            <span>{product.category_name}</span>
            <span>од {product.seller_name}</span>
          </div>
        </div>
      </Link>
      
      {/* Add to Cart Section - Not part of the link */}
      <div className="px-4 pb-4">
        <div className="flex items-center justify-between">
          <div className="text-sm text-gray-500">
            {product.stock_quantity > 0 ? (
              <span className="text-green-600">
                ✓ {product.stock_quantity} на залиха
              </span>
            ) : (
              <span className="text-red-600">✗ Нема на залиха</span>
            )}
          </div>
          
          {isAuthenticated() && (
            <button
              onClick={handleAddToCart}
              disabled={isAddingToCart || product.stock_quantity < 1}
              className={`px-4 py-2 rounded-md text-sm font-medium transition-colors duration-200 ${
                product.stock_quantity < 1
                  ? 'bg-gray-200 text-gray-500 cursor-not-allowed'
                  : isAddingToCart
                  ? 'bg-blue-400 text-white cursor-not-allowed'
                  : 'bg-blue-600 text-white hover:bg-blue-700'
              }`}
            >
              {isAddingToCart ? (
                <div className="flex items-center space-x-2">
                  <div className="w-4 h-4 animate-spin border-2 border-white border-t-transparent rounded-full"></div>
                  <span>Adding...</span>
                </div>
              ) : (
                '+ Додај во кошничка'
              )}
            </button>
          )}
        </div>
      </div>
    </div>
  );
};

export default ProductCard;
