import React, { useState } from 'react';
import { useAuth } from '../context/AuthContext';
import StarRating from './StarRating';
import api from '../services/api';

const ReviewForm = ({ productId, onReviewAdded, onCancel }) => {
    const { user } = useAuth();
    const [formData, setFormData] = useState({
        rating: 0,
        comment: ''
    });
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);

    const handleSubmit = async (e) => {
        e.preventDefault();
        
        if (formData.rating === 0) {
            setError('Please select a rating');
            return;
        }

        if (!formData.comment.trim()) {
            setError('Please write a comment');
            return;
        }

        setLoading(true);
        setError(null);

        try {
            const response = await api.post('/reviews', {
                product_id: productId,
                rating: formData.rating,
                comment: formData.comment.trim()
            });

            if (response.data.success) {
                setFormData({ rating: 0, comment: '' });
                if (onReviewAdded) {
                    onReviewAdded(response.data.review);
                }
            } else {
                setError(response.data.message || 'Failed to submit review');
            }
        } catch (error) {
            console.error('Error submitting review:', error);
            setError('Failed to submit review. Please try again.');
        } finally {
            setLoading(false);
        }
    };

    const handleRatingChange = (rating) => {
        setFormData(prev => ({ ...prev, rating }));
        setError(null);
    };

    const handleCommentChange = (e) => {
        setFormData(prev => ({ ...prev, comment: e.target.value }));
        setError(null);
    };

    if (!user) {
        return (
            <div className="bg-gray-100 p-6 rounded-lg text-center">
                <p className="text-gray-600 mb-4">Please log in to write a review</p>
                <a
                    href="/login"
                    className="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700"
                >
                    Log In
                </a>
            </div>
        );
    }

    return (
        <div className="bg-white p-6 rounded-lg border border-gray-200">
            <h3 className="text-lg font-semibold text-gray-800 mb-4">Write a Review</h3>
            
            <form onSubmit={handleSubmit}>
                {/* Rating */}
                <div className="mb-4">
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                        Rating *
                    </label>
                    <StarRating
                        rating={formData.rating}
                        onRatingChange={handleRatingChange}
                        readonly={false}
                        showRatingText={false}
                        size="lg"
                    />
                </div>

                {/* Comment */}
                <div className="mb-4">
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                        Your Review *
                    </label>
                    <textarea
                        rows={4}
                        value={formData.comment}
                        onChange={handleCommentChange}
                        placeholder="Share your thoughts about this product..."
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
                        disabled={loading}
                    />
                    <p className="text-xs text-gray-500 mt-1">
                        {formData.comment.length}/500 characters
                    </p>
                </div>

                {/* Error Message */}
                {error && (
                    <div className="mb-4 p-3 bg-red-100 border border-red-300 text-red-700 rounded-md">
                        {error}
                    </div>
                )}

                {/* Buttons */}
                <div className="flex space-x-3">
                    <button
                        type="submit"
                        disabled={loading}
                        className="flex-1 bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                        {loading ? 'Submitting...' : 'Submit Review'}
                    </button>
                    
                    {onCancel && (
                        <button
                            type="button"
                            onClick={onCancel}
                            disabled={loading}
                            className="flex-1 border border-gray-300 text-gray-700 py-2 px-4 rounded-md hover:bg-gray-50 disabled:opacity-50"
                        >
                            Cancel
                        </button>
                    )}
                </div>
            </form>
        </div>
    );
};

export default ReviewForm;