import React, { useState } from 'react';
import { useAuth } from '../context/AuthContext';
import StarRating from './StarRating';
import api from '../services/api';

const ReviewList = ({ reviews, onReviewUpdate, onReviewDelete }) => {
    const { user } = useAuth();
    const [editingReview, setEditingReview] = useState(null);
    const [editFormData, setEditFormData] = useState({ rating: 0, comment: '' });
    const [loading, setLoading] = useState(false);

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
    };

    const handleEditStart = (review) => {
        setEditingReview(review.id);
        setEditFormData({
            rating: review.rating,
            comment: review.comment
        });
    };

    const handleEditCancel = () => {
        setEditingReview(null);
        setEditFormData({ rating: 0, comment: '' });
    };

    const handleEditSubmit = async (reviewId) => {
        setLoading(true);
        try {
            const response = await api.put(`/reviews/${reviewId}`, {
                rating: editFormData.rating,
                comment: editFormData.comment
            });

            if (response.data.success) {
                if (onReviewUpdate) {
                    onReviewUpdate(response.data.review);
                }
                setEditingReview(null);
            }
        } catch (error) {
            console.error('Error updating review:', error);
            alert('Failed to update review');
        } finally {
            setLoading(false);
        }
    };

    const handleDelete = async (reviewId) => {
        if (!window.confirm('Are you sure you want to delete this review?')) {
            return;
        }

        try {
            const response = await api.delete(`/reviews/${reviewId}`);
            
            if (response.data.success) {
                if (onReviewDelete) {
                    onReviewDelete(reviewId);
                }
            }
        } catch (error) {
            console.error('Error deleting review:', error);
            alert('Failed to delete review');
        }
    };

    const handleHelpful = async (reviewId, isHelpful) => {
        try {
            await api.post(`/reviews/${reviewId}/helpful`, {
                is_helpful: isHelpful
            });
            // You might want to refresh the reviews here or update the helpful count
        } catch (error) {
            console.error('Error marking review helpful:', error);
        }
    };

    if (!reviews || reviews.length === 0) {
        return (
            <div className="bg-gray-50 p-8 rounded-lg text-center">
                <div className="mb-4">
                    <svg className="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 8h10M7 12h4m1 8l-4-4H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-1l-4 4z" />
                    </svg>
                </div>
                <p className="text-gray-600">No reviews yet. Be the first to review this product!</p>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {reviews.map((review) => (
                <div key={review.id} className="bg-white p-6 rounded-lg border border-gray-200">
                    {/* Review Header */}
                    <div className="flex items-start justify-between mb-4">
                        <div className="flex items-center space-x-3">
                            <div className="w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center">
                                <span className="text-white font-medium">
                                    {review.user_first_name ? review.user_first_name.charAt(0).toUpperCase() : 'U'}
                                </span>
                            </div>
                            <div>
                                <p className="font-medium text-gray-800">
                                    {review.user_first_name} {review.user_last_name}
                                </p>
                                <p className="text-sm text-gray-500">
                                    {formatDate(review.created_at)}
                                </p>
                            </div>
                        </div>

                        {/* Review Actions - only show for own reviews */}
                        {user && user.id === review.user_id && (
                            <div className="flex space-x-2">
                                <button
                                    onClick={() => handleEditStart(review)}
                                    className="text-sm text-blue-600 hover:text-blue-700"
                                >
                                    Edit
                                </button>
                                <button
                                    onClick={() => handleDelete(review.id)}
                                    className="text-sm text-red-600 hover:text-red-700"
                                >
                                    Delete
                                </button>
                            </div>
                        )}
                    </div>

                    {/* Rating */}
                    <div className="mb-3">
                        <StarRating 
                            rating={editingReview === review.id ? editFormData.rating : review.rating}
                            onRatingChange={editingReview === review.id ? 
                                (rating) => setEditFormData(prev => ({ ...prev, rating })) : 
                                null
                            }
                            readonly={editingReview !== review.id}
                            showRatingText={false}
                        />
                    </div>

                    {/* Review Content */}
                    {editingReview === review.id ? (
                        <div className="mb-4">
                            <textarea
                                value={editFormData.comment}
                                onChange={(e) => setEditFormData(prev => ({ ...prev, comment: e.target.value }))}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
                                rows={3}
                                disabled={loading}
                            />
                            <div className="flex space-x-2 mt-3">
                                <button
                                    onClick={() => handleEditSubmit(review.id)}
                                    disabled={loading}
                                    className="bg-blue-600 text-white px-3 py-1 rounded-md text-sm hover:bg-blue-700 disabled:opacity-50"
                                >
                                    {loading ? 'Saving...' : 'Save'}
                                </button>
                                <button
                                    onClick={handleEditCancel}
                                    disabled={loading}
                                    className="border border-gray-300 text-gray-700 px-3 py-1 rounded-md text-sm hover:bg-gray-50 disabled:opacity-50"
                                >
                                    Cancel
                                </button>
                            </div>
                        </div>
                    ) : (
                        <p className="text-gray-700 mb-4">{review.comment}</p>
                    )}

                    {/* Helpful Section */}
                    {editingReview !== review.id && (
                        <div className="flex items-center space-x-4 text-sm text-gray-500">
                            <span>Was this review helpful?</span>
                            <div className="flex space-x-2">
                                <button
                                    onClick={() => handleHelpful(review.id, true)}
                                    className="flex items-center space-x-1 hover:text-green-600"
                                >
                                    <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14 10h4.764a2 2 0 011.789 2.894l-3.5 7A2 2 0 0115.263 21h-4.017c-.163 0-.326-.02-.485-.06L7 20m7-10V5a2 2 0 00-2-2h-.095c-.5 0-.905.405-.905.905 0 .714-.211 1.412-.608 2.006L9 7v3m-3 10v-1a2 2 0 012-2h1m-6 3a2 2 0 002-2v-7a2 2 0 00-2-2H3a2 2 0 00-2 2v9a2 2 0 002 2z" />
                                    </svg>
                                    <span>Yes</span>
                                    {review.helpful_count > 0 && (
                                        <span>({review.helpful_count})</span>
                                    )}
                                </button>
                                <button
                                    onClick={() => handleHelpful(review.id, false)}
                                    className="flex items-center space-x-1 hover:text-red-600"
                                >
                                    <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 14H5.236a2 2 0 01-1.789-2.894l3.5-7A2 2 0 018.736 3h4.018c.163 0 .326.02.485.06L17 4m-7 10v2a2 2 0 002 2h.095c.5 0 .905-.405.905-.905 0-.714.211-1.412.608-2.006L15 17v-3m-6-10.5a2 2 0 00-2 2v1a2 2 0 002 2h1m-1 4v-1a2 2 0 012-2h1" />
                                    </svg>
                                    <span>No</span>
                                    {review.not_helpful_count > 0 && (
                                        <span>({review.not_helpful_count})</span>
                                    )}
                                </button>
                            </div>
                        </div>
                    )}
                </div>
            ))}
        </div>
    );
};

export default ReviewList;