import React from 'react';

const StarRating = ({ 
    rating, 
    onRatingChange, 
    readonly = false, 
    size = 'md',
    showRatingText = true 
}) => {
    const sizes = {
        sm: 'w-4 h-4',
        md: 'w-5 h-5',
        lg: 'w-6 h-6'
    };

    const handleStarClick = (starValue) => {
        if (!readonly && onRatingChange) {
            onRatingChange(starValue);
        }
    };

    const renderStar = (starValue) => {
        const isFilled = starValue <= rating;
        const isHalfFilled = starValue - 0.5 === rating;
        
        return (
            <button
                key={starValue}
                type="button"
                disabled={readonly}
                onClick={() => handleStarClick(starValue)}
                className={`${sizes[size]} ${
                    readonly 
                        ? 'cursor-default' 
                        : 'cursor-pointer hover:scale-110 transition-transform'
                } focus:outline-none`}
            >
                <svg
                    className={`${sizes[size]} ${
                        isFilled 
                            ? 'text-yellow-400' 
                            : isHalfFilled 
                                ? 'text-yellow-400' 
                                : 'text-gray-300'
                    }`}
                    fill="currentColor"
                    viewBox="0 0 20 20"
                >
                    <path
                        d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"
                    />
                    {isHalfFilled && (
                        <defs>
                            <linearGradient id={`half-${starValue}`}>
                                <stop offset="50%" stopColor="currentColor" />
                                <stop offset="50%" stopColor="#D1D5DB" />
                            </linearGradient>
                        </defs>
                    )}
                </svg>
            </button>
        );
    };

    return (
        <div className="flex items-center space-x-1">
            <div className="flex items-center">
                {[1, 2, 3, 4, 5].map(starValue => renderStar(starValue))}
            </div>
            
            {showRatingText && (
                <span className="text-sm text-gray-600 ml-2">
                    {rating > 0 ? `${rating.toFixed(1)} out of 5` : 'No rating'}
                </span>
            )}
        </div>
    );
};

export default StarRating;