import React, { createContext, useContext, useReducer, useEffect } from 'react';
import { useAuth } from './AuthContext';
import api from '../services/api';

const CartContext = createContext();

const cartReducer = (state, action) => {
    switch (action.type) {
        case 'SET_CART':
            return {
                ...state,
                items: action.payload.cartItems || [],
                totalItems: action.payload.totalItems || 0,
                totalPrice: action.payload.totalPrice || 0,
                loading: false
            };
        
        case 'SET_LOADING':
            return {
                ...state,
                loading: action.payload
            };
        
        case 'ADD_TO_CART':
            const existingItemIndex = state.items.findIndex(
                item => item.product_id === action.payload.product_id
            );
            
            if (existingItemIndex >= 0) {
                const updatedItems = [...state.items];
                updatedItems[existingItemIndex].quantity += action.payload.quantity;
                return {
                    ...state,
                    items: updatedItems,
                    totalItems: state.totalItems + action.payload.quantity,
                    totalPrice: state.totalPrice + (action.payload.price * action.payload.quantity)
                };
            } else {
                return {
                    ...state,
                    items: [...state.items, action.payload],
                    totalItems: state.totalItems + action.payload.quantity,
                    totalPrice: state.totalPrice + (action.payload.price * action.payload.quantity)
                };
            }
        
        case 'UPDATE_CART_ITEM':
            const updatedItems = state.items.map(item => 
                item.id === action.payload.id 
                    ? { ...item, quantity: action.payload.quantity }
                    : item
            );
            
            const newTotalItems = updatedItems.reduce((sum, item) => sum + item.quantity, 0);
            const newTotalPrice = updatedItems.reduce((sum, item) => sum + (parseFloat(item.price) * item.quantity), 0);
            
            return {
                ...state,
                items: updatedItems,
                totalItems: newTotalItems,
                totalPrice: newTotalPrice
            };
        
        case 'REMOVE_FROM_CART':
            const filteredItems = state.items.filter(item => item.id !== action.payload.id);
            const removedItem = state.items.find(item => item.id === action.payload.id);
            
            return {
                ...state,
                items: filteredItems,
                totalItems: state.totalItems - (removedItem?.quantity || 0),
                totalPrice: state.totalPrice - (parseFloat(removedItem?.price || 0) * (removedItem?.quantity || 0))
            };
        
        case 'CLEAR_CART':
            return {
                ...state,
                items: [],
                totalItems: 0,
                totalPrice: 0
            };
        
        case 'SET_ERROR':
            return {
                ...state,
                error: action.payload,
                loading: false
            };
        
        case 'CLEAR_ERROR':
            return {
                ...state,
                error: null
            };
        
        default:
            return state;
    }
};

const initialState = {
    items: [],
    totalItems: 0,
    totalPrice: 0,
    loading: false,
    error: null
};

export const CartProvider = ({ children }) => {
    const [state, dispatch] = useReducer(cartReducer, initialState);
    const { user } = useAuth();

    // Load cart when user logs in
    useEffect(() => {
        const token = localStorage.getItem('token');
        if (user && token) {
            loadCart();
        } else {
            // Clear cart when user logs out
            dispatch({ type: 'CLEAR_CART' });
        }
    }, [user]);

    const loadCart = async () => {
        try {
            dispatch({ type: 'SET_LOADING', payload: true });
            const response = await api.get('/cart');
            dispatch({ type: 'SET_CART', payload: response.data });
        } catch (error) {
            console.error('Error loading cart:', error);
            dispatch({ type: 'SET_ERROR', payload: 'Failed to load cart' });
        }
    };

    const addToCart = async (productId, quantity = 1) => {
        try {
            dispatch({ type: 'CLEAR_ERROR' });
            const response = await api.post('/cart/add', { productId, quantity });
            
            if (response.data.success) {
                // Reload cart to get updated data
                await loadCart();
                return { success: true, message: response.data.message };
            }
        } catch (error) {
            const errorMessage = error.response?.data?.message || 'Failed to add to cart';
            dispatch({ type: 'SET_ERROR', payload: errorMessage });
            return { success: false, message: errorMessage };
        }
    };

    const updateCartItem = async (cartItemId, quantity) => {
        try {
            dispatch({ type: 'CLEAR_ERROR' });
            const response = await api.put(`/cart/update/${cartItemId}`, { quantity });
            
            if (response.data.success) {
                dispatch({ 
                    type: 'UPDATE_CART_ITEM', 
                    payload: { id: cartItemId, quantity } 
                });
                return { success: true, message: response.data.message };
            }
        } catch (error) {
            const errorMessage = error.response?.data?.message || 'Failed to update cart';
            dispatch({ type: 'SET_ERROR', payload: errorMessage });
            return { success: false, message: errorMessage };
        }
    };

    const removeFromCart = async (cartItemId) => {
        try {
            dispatch({ type: 'CLEAR_ERROR' });
            const response = await api.delete(`/cart/remove/${cartItemId}`);
            
            if (response.data.success) {
                dispatch({ type: 'REMOVE_FROM_CART', payload: { id: cartItemId } });
                return { success: true, message: response.data.message };
            }
        } catch (error) {
            const errorMessage = error.response?.data?.message || 'Failed to remove from cart';
            dispatch({ type: 'SET_ERROR', payload: errorMessage });
            return { success: false, message: errorMessage };
        }
    };

    const clearCart = async () => {
        try {
            dispatch({ type: 'CLEAR_ERROR' });
            const response = await api.delete('/cart/clear');
            
            if (response.data.success) {
                dispatch({ type: 'CLEAR_CART' });
                return { success: true, message: response.data.message };
            }
        } catch (error) {
            const errorMessage = error.response?.data?.message || 'Failed to clear cart';
            dispatch({ type: 'SET_ERROR', payload: errorMessage });
            return { success: false, message: errorMessage };
        }
    };

    const getCartSummary = async () => {
        try {
            const response = await api.get('/cart/summary');
            return response.data.summary;
        } catch (error) {
            console.error('Error getting cart summary:', error);
            return { totalItems: 0, totalQuantity: 0, totalPrice: 0 };
        }
    };

    const value = {
        // State
        cartItems: state.items,
        totalItems: state.totalItems,
        totalPrice: state.totalPrice,
        loading: state.loading,
        error: state.error,
        
        // Actions
        loadCart,
        addToCart,
        updateCartItem,
        removeFromCart,
        clearCart,
        getCartSummary,
        clearError: () => dispatch({ type: 'CLEAR_ERROR' })
    };

    return (
        <CartContext.Provider value={value}>
            {children}
        </CartContext.Provider>
    );
};

export const useCart = () => {
    const context = useContext(CartContext);
    if (!context) {
        throw new Error('useCart must be used within a CartProvider');
    }
    return context;
};