#!/usr/bin/env node

const { Pool } = require('pg');
const fs = require('fs');
const path = require('path');

// Database configuration
const pool = new Pool({
  host: process.env.DB_HOST || 'localhost',
  port: process.env.DB_PORT || 5432,
  database: process.env.DB_NAME || 'kupi_mk',
  user: process.env.DB_USER || 'admin',
  password: process.env.DB_PASSWORD || 'password123',
});

async function exportDatabase() {
  const client = await pool.connect();
  
  try {
    console.log('🔄 Starting database export...');
    
    // Create exports directory if it doesn't exist
    const exportsDir = path.join(__dirname, '../database-exports');
    if (!fs.existsSync(exportsDir)) {
      fs.mkdirSync(exportsDir, { recursive: true });
    }
    
    // Export schema (table structure)
    console.log('📋 Exporting schema...');
    
    let schemaSQL = "-- Database Schema Export\\n-- Generated on: " + new Date().toISOString() + "\\n\\n";
    
    // Add the original schema from database_setup.sql
    const setupSQL = fs.readFileSync(path.join(__dirname, '../database_setup.sql'), 'utf8');
    schemaSQL += setupSQL + "\\n\\n";
    
    // Export data
    console.log('📊 Exporting data...');
    
    // Export users
    const usersResult = await client.query('SELECT * FROM users ORDER BY id');
    schemaSQL += "-- Users Data\\n";
    schemaSQL += "DELETE FROM users;\\n";
    for (const user of usersResult.rows) {
      const values = [
        user.id,
        `'${user.username.replace(/'/g, "''")}'`,
        `'${user.email.replace(/'/g, "''")}'`,
        `'${user.password.replace(/'/g, "''")}'`,
        `'${user.first_name.replace(/'/g, "''")}'`,
        `'${user.last_name.replace(/'/g, "''")}'`,
        user.phone ? `'${user.phone.replace(/'/g, "''")}'` : 'NULL',
        user.address ? `'${user.address.replace(/'/g, "''")}'` : 'NULL',
        user.is_seller,
        `'${user.created_at.toISOString()}'`,
        `'${user.updated_at.toISOString()}'`
      ].join(', ');
      
      schemaSQL += `INSERT INTO users (id, username, email, password, first_name, last_name, phone, address, is_seller, created_at, updated_at) VALUES (${values});\\n`;
    }
    
    // Export categories
    const categoriesResult = await client.query('SELECT * FROM categories ORDER BY id');
    schemaSQL += "\\n-- Categories Data\\n";
    schemaSQL += "DELETE FROM categories;\\n";
    for (const category of categoriesResult.rows) {
      const values = [
        category.id,
        `'${category.name.replace(/'/g, "''")}'`,
        category.description ? `'${category.description.replace(/'/g, "''")}'` : 'NULL',
        `'${category.created_at.toISOString()}'`
      ].join(', ');
      
      schemaSQL += `INSERT INTO categories (id, name, description, created_at) VALUES (${values});\\n`;
    }
    
    // Export products
    const productsResult = await client.query('SELECT * FROM products ORDER BY id');
    schemaSQL += "\\n-- Products Data\\n";
    schemaSQL += "DELETE FROM products;\\n";
    for (const product of productsResult.rows) {
      const imagesArray = product.images ? `ARRAY[${product.images.map(img => `'${img.replace(/'/g, "''")}'`).join(', ')}]` : 'NULL';
      const values = [
        product.id,
        `'${product.title.replace(/'/g, "''")}'`,
        `'${product.description.replace(/'/g, "''")}'`,
        product.price,
        product.category_id || 'NULL',
        product.seller_id,
        imagesArray,
        product.stock_quantity || 0,
        product.location ? `'${product.location.replace(/'/g, "''")}'` : 'NULL',
        product.is_active,
        `'${product.created_at.toISOString()}'`,
        `'${product.updated_at.toISOString()}'`
      ].join(', ');
      
      schemaSQL += `INSERT INTO products (id, title, description, price, category_id, seller_id, images, stock_quantity, location, is_active, created_at, updated_at) VALUES (${values});\\n`;
    }
    
    // Reset sequences
    schemaSQL += "\\n-- Reset sequences\\n";
    schemaSQL += "SELECT setval('users_id_seq', (SELECT MAX(id) FROM users));\\n";
    schemaSQL += "SELECT setval('categories_id_seq', (SELECT MAX(id) FROM categories));\\n";
    schemaSQL += "SELECT setval('products_id_seq', (SELECT MAX(id) FROM products));\\n";
    
    // Write to file
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
    const filename = `database-export-${timestamp}.sql`;
    const filepath = path.join(exportsDir, filename);
    
    fs.writeFileSync(filepath, schemaSQL);
    
    // Also create a latest export for easy access
    fs.writeFileSync(path.join(exportsDir, 'latest-export.sql'), schemaSQL);
    
    console.log('✅ Database exported successfully!');
    console.log(`📁 Export saved to: ${filepath}`);
    console.log(`📁 Latest export: ${path.join(exportsDir, 'latest-export.sql')}`);
    
    // Show statistics
    console.log('\\n📊 Export Statistics:');
    console.log(`👥 Users: ${usersResult.rows.length}`);
    console.log(`📂 Categories: ${categoriesResult.rows.length}`);
    console.log(`📦 Products: ${productsResult.rows.length}`);
    
  } catch (error) {
    console.error('❌ Export failed:', error);
  } finally {
    client.release();
    await pool.end();
  }
}

// Run export if called directly
if (require.main === module) {
  exportDatabase();
}

module.exports = { exportDatabase };
