const express = require('express');
const router = express.Router();
const { pool } = require('../config/database');
const { auth } = require('../middleware/auth');

// Create new order from cart (checkout)
router.post('/checkout', auth, async (req, res) => {
    const client = await pool.connect();
    
    try {
        await client.query('BEGIN');
        
        const userId = req.user.id;
        const { 
            shippingAddress, 
            billingAddress, 
            phone, 
            paymentMethod = 'cash_on_delivery',
            notes 
        } = req.body;
        
        if (!shippingAddress || !phone) {
            return res.status(400).json({ 
                success: false, 
                message: 'Shipping address and phone are required' 
            });
        }
        
        // Get cart items with current prices
        const cartResult = await client.query(`
            SELECT 
                ci.id as cart_item_id,
                ci.product_id,
                ci.quantity,
                p.title,
                p.price,
                p.stock_quantity,
                p.is_active
            FROM cart_items ci
            JOIN products p ON ci.product_id = p.id
            WHERE ci.user_id = $1 AND p.is_active = true
        `, [userId]);
        
        if (cartResult.rows.length === 0) {
            await client.query('ROLLBACK');
            return res.status(400).json({ 
                success: false, 
                message: 'Cart is empty' 
            });
        }
        
        const cartItems = cartResult.rows;
        
        // Validate stock availability
        for (const item of cartItems) {
            if (item.quantity > item.stock_quantity) {
                await client.query('ROLLBACK');
                return res.status(400).json({ 
                    success: false, 
                    message: `Insufficient stock for ${item.title}. Only ${item.stock_quantity} available.` 
                });
            }
        }
        
        // Calculate total amount
        const totalAmount = cartItems.reduce((sum, item) => sum + (parseFloat(item.price) * item.quantity), 0);
        
        // Create order
        const orderResult = await client.query(`
            INSERT INTO orders (user_id, total_amount, shipping_address, billing_address, phone, payment_method, notes)
            VALUES ($1, $2, $3, $4, $5, $6, $7)
            RETURNING *
        `, [userId, totalAmount, shippingAddress, billingAddress || shippingAddress, phone, paymentMethod, notes]);
        
        const order = orderResult.rows[0];
        
        // Create order items and update stock
        for (const item of cartItems) {
            // Add to order_items
            await client.query(`
                INSERT INTO order_items (order_id, product_id, quantity, price_at_time)
                VALUES ($1, $2, $3, $4)
            `, [order.id, item.product_id, item.quantity, item.price]);
            
            // Update product stock
            await client.query(`
                UPDATE products 
                SET stock_quantity = stock_quantity - $1 
                WHERE id = $2
            `, [item.quantity, item.product_id]);
        }
        
        // Clear cart
        await client.query('DELETE FROM cart_items WHERE user_id = $1', [userId]);
        
        await client.query('COMMIT');
        
        res.json({
            success: true,
            message: 'Order placed successfully',
            order: {
                id: order.id,
                totalAmount: order.total_amount,
                status: order.status,
                createdAt: order.created_at
            }
        });
        
    } catch (error) {
        await client.query('ROLLBACK');
        console.error('Error during checkout:', error);
        res.status(500).json({ success: false, message: 'Server error during checkout' });
    } finally {
        client.release();
    }
});

// Get user's order history
router.get('/history', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const page = parseInt(req.query.page) || 1;
        const limit = parseInt(req.query.limit) || 10;
        const offset = (page - 1) * limit;
        
        const result = await pool.query(`
            SELECT 
                o.*,
                COUNT(oi.id) as item_count
            FROM orders o
            LEFT JOIN order_items oi ON o.id = oi.order_id
            WHERE o.user_id = $1
            GROUP BY o.id
            ORDER BY o.created_at DESC
            LIMIT $2 OFFSET $3
        `, [userId, limit, offset]);
        
        // Get total count for pagination
        const countResult = await pool.query(
            'SELECT COUNT(*) FROM orders WHERE user_id = $1',
            [userId]
        );
        
        res.json({
            success: true,
            orders: result.rows,
            pagination: {
                page,
                limit,
                totalOrders: parseInt(countResult.rows[0].count),
                totalPages: Math.ceil(countResult.rows[0].count / limit)
            }
        });
    } catch (error) {
        console.error('Error fetching order history:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Get specific order details
router.get('/:orderId', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const orderId = req.params.orderId;
        
        // Get order details
        const orderResult = await pool.query(`
            SELECT * FROM orders 
            WHERE id = $1 AND user_id = $2
        `, [orderId, userId]);
        
        if (orderResult.rows.length === 0) {
            return res.status(404).json({ success: false, message: 'Order not found' });
        }
        
        const order = orderResult.rows[0];
        
        // Get order items
        const itemsResult = await pool.query(`
            SELECT 
                oi.*,
                p.title,
                p.images,
                p.description
            FROM order_items oi
            JOIN products p ON oi.product_id = p.id
            WHERE oi.order_id = $1
            ORDER BY oi.created_at
        `, [orderId]);
        
        res.json({
            success: true,
            order: {
                ...order,
                items: itemsResult.rows
            }
        });
    } catch (error) {
        console.error('Error fetching order details:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Update order status (for sellers/admins - will implement later)
router.put('/:orderId/status', auth, async (req, res) => {
    try {
        const orderId = req.params.orderId;
        const { status } = req.body;
        
        const validStatuses = ['pending', 'confirmed', 'processing', 'shipped', 'delivered', 'cancelled'];
        
        if (!validStatuses.includes(status)) {
            return res.status(400).json({ 
                success: false, 
                message: 'Invalid status. Valid statuses: ' + validStatuses.join(', ') 
            });
        }
        
        // For now, only allow users to cancel their own pending orders
        // Later we can add role-based permissions for sellers/admins
        const userId = req.user.id;
        
        if (status === 'cancelled') {
            const result = await pool.query(`
                UPDATE orders 
                SET status = $1, updated_at = CURRENT_TIMESTAMP 
                WHERE id = $2 AND user_id = $3 AND status = 'pending'
                RETURNING *
            `, [status, orderId, userId]);
            
            if (result.rows.length === 0) {
                return res.status(400).json({ 
                    success: false, 
                    message: 'Order cannot be cancelled or not found' 
                });
            }
            
            // Restore stock quantities when order is cancelled
            await pool.query(`
                UPDATE products 
                SET stock_quantity = stock_quantity + oi.quantity
                FROM order_items oi 
                WHERE products.id = oi.product_id AND oi.order_id = $1
            `, [orderId]);
            
            res.json({
                success: true,
                message: 'Order cancelled successfully',
                order: result.rows[0]
            });
        } else {
            res.status(403).json({ 
                success: false, 
                message: 'Unauthorized to update order status' 
            });
        }
    } catch (error) {
        console.error('Error updating order status:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Get order statistics (for dashboard)
router.get('/stats/summary', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        
        const result = await pool.query(`
            SELECT 
                COUNT(*) as total_orders,
                COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_orders,
                COUNT(CASE WHEN status = 'delivered' THEN 1 END) as delivered_orders,
                COALESCE(SUM(total_amount), 0) as total_spent,
                COALESCE(SUM(CASE WHEN status = 'delivered' THEN total_amount ELSE 0 END), 0) as total_completed
            FROM orders 
            WHERE user_id = $1
        `, [userId]);
        
        res.json({
            success: true,
            stats: result.rows[0]
        });
    } catch (error) {
        console.error('Error fetching order stats:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

module.exports = router;