const express = require('express');
const router = express.Router();
const { pool } = require('../config/database');
const { auth } = require('../middleware/auth');

// Get reviews for a product
router.get('/product/:productId', async (req, res) => {
    try {
        const productId = req.params.productId;
        const page = parseInt(req.query.page) || 1;
        const limit = parseInt(req.query.limit) || 10;
        const sortBy = req.query.sortBy || 'newest'; // newest, oldest, rating_high, rating_low, helpful
        const offset = (page - 1) * limit;
        
        let orderClause = 'pr.created_at DESC';
        
        switch (sortBy) {
            case 'oldest':
                orderClause = 'pr.created_at ASC';
                break;
            case 'rating_high':
                orderClause = 'pr.rating DESC, pr.created_at DESC';
                break;
            case 'rating_low':
                orderClause = 'pr.rating ASC, pr.created_at DESC';
                break;
            case 'helpful':
                orderClause = 'pr.helpful_votes DESC, pr.created_at DESC';
                break;
            default:
                orderClause = 'pr.created_at DESC';
        }
        
        const result = await pool.query(`
            SELECT 
                pr.*,
                u.username,
                u.first_name,
                u.last_name
            FROM product_reviews pr
            JOIN users u ON pr.user_id = u.id
            WHERE pr.product_id = $1
            ORDER BY ${orderClause}
            LIMIT $2 OFFSET $3
        `, [productId, limit, offset]);
        
        // Get review statistics
        const statsResult = await pool.query(`
            SELECT 
                COUNT(*) as total_reviews,
                AVG(rating) as average_rating,
                COUNT(CASE WHEN rating = 5 THEN 1 END) as five_star,
                COUNT(CASE WHEN rating = 4 THEN 1 END) as four_star,
                COUNT(CASE WHEN rating = 3 THEN 1 END) as three_star,
                COUNT(CASE WHEN rating = 2 THEN 1 END) as two_star,
                COUNT(CASE WHEN rating = 1 THEN 1 END) as one_star
            FROM product_reviews
            WHERE product_id = $1
        `, [productId]);
        
        const stats = statsResult.rows[0];
        
        res.json({
            success: true,
            reviews: result.rows,
            statistics: {
                totalReviews: parseInt(stats.total_reviews),
                averageRating: parseFloat(stats.average_rating) || 0,
                ratingDistribution: {
                    5: parseInt(stats.five_star),
                    4: parseInt(stats.four_star),
                    3: parseInt(stats.three_star),
                    2: parseInt(stats.two_star),
                    1: parseInt(stats.one_star)
                }
            },
            pagination: {
                page,
                limit,
                totalPages: Math.ceil(stats.total_reviews / limit)
            }
        });
    } catch (error) {
        console.error('Error fetching product reviews:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Add a review for a product
router.post('/add', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const { productId, rating, title, comment } = req.body;
        
        if (!productId || !rating) {
            return res.status(400).json({ 
                success: false, 
                message: 'Product ID and rating are required' 
            });
        }
        
        if (rating < 1 || rating > 5) {
            return res.status(400).json({ 
                success: false, 
                message: 'Rating must be between 1 and 5' 
            });
        }
        
        // Check if product exists
        const productResult = await pool.query(
            'SELECT id, title FROM products WHERE id = $1 AND is_active = true',
            [productId]
        );
        
        if (productResult.rows.length === 0) {
            return res.status(404).json({ success: false, message: 'Product not found' });
        }
        
        // Check if user already reviewed this product
        const existingReview = await pool.query(
            'SELECT id FROM product_reviews WHERE product_id = $1 AND user_id = $2',
            [productId, userId]
        );
        
        if (existingReview.rows.length > 0) {
            return res.status(400).json({ 
                success: false, 
                message: 'You have already reviewed this product' 
            });
        }
        
        // Check if user has purchased this product (for verified purchase)
        const purchaseResult = await pool.query(`
            SELECT oi.id FROM order_items oi
            JOIN orders o ON oi.order_id = o.id
            WHERE o.user_id = $1 AND oi.product_id = $2 AND o.status IN ('delivered', 'confirmed')
        `, [userId, productId]);
        
        const verifiedPurchase = purchaseResult.rows.length > 0;
        
        const result = await pool.query(`
            INSERT INTO product_reviews (product_id, user_id, rating, title, comment, verified_purchase)
            VALUES ($1, $2, $3, $4, $5, $6)
            RETURNING *
        `, [productId, userId, rating, title, comment, verifiedPurchase]);
        
        res.json({
            success: true,
            message: 'Review added successfully',
            review: result.rows[0]
        });
    } catch (error) {
        console.error('Error adding review:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Update a review
router.put('/:reviewId', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const reviewId = req.params.reviewId;
        const { rating, title, comment } = req.body;
        
        if (!rating) {
            return res.status(400).json({ 
                success: false, 
                message: 'Rating is required' 
            });
        }
        
        if (rating < 1 || rating > 5) {
            return res.status(400).json({ 
                success: false, 
                message: 'Rating must be between 1 and 5' 
            });
        }
        
        const result = await pool.query(`
            UPDATE product_reviews 
            SET rating = $1, title = $2, comment = $3, updated_at = CURRENT_TIMESTAMP
            WHERE id = $4 AND user_id = $5
            RETURNING *
        `, [rating, title, comment, reviewId, userId]);
        
        if (result.rows.length === 0) {
            return res.status(404).json({ 
                success: false, 
                message: 'Review not found or unauthorized' 
            });
        }
        
        res.json({
            success: true,
            message: 'Review updated successfully',
            review: result.rows[0]
        });
    } catch (error) {
        console.error('Error updating review:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Delete a review
router.delete('/:reviewId', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const reviewId = req.params.reviewId;
        
        const result = await pool.query(
            'DELETE FROM product_reviews WHERE id = $1 AND user_id = $2 RETURNING *',
            [reviewId, userId]
        );
        
        if (result.rows.length === 0) {
            return res.status(404).json({ 
                success: false, 
                message: 'Review not found or unauthorized' 
            });
        }
        
        res.json({
            success: true,
            message: 'Review deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting review:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Vote on review helpfulness
router.post('/:reviewId/helpful', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const reviewId = req.params.reviewId;
        const { isHelpful } = req.body; // true or false
        
        if (typeof isHelpful !== 'boolean') {
            return res.status(400).json({ 
                success: false, 
                message: 'isHelpful must be true or false' 
            });
        }
        
        // Check if review exists
        const reviewResult = await pool.query(
            'SELECT id, user_id FROM product_reviews WHERE id = $1',
            [reviewId]
        );
        
        if (reviewResult.rows.length === 0) {
            return res.status(404).json({ success: false, message: 'Review not found' });
        }
        
        // Don't allow users to vote on their own reviews
        if (reviewResult.rows[0].user_id === userId) {
            return res.status(400).json({ 
                success: false, 
                message: 'Cannot vote on your own review' 
            });
        }
        
        // Check if user already voted
        const existingVote = await pool.query(
            'SELECT id, is_helpful FROM review_helpfulness WHERE review_id = $1 AND user_id = $2',
            [reviewId, userId]
        );
        
        if (existingVote.rows.length > 0) {
            // Update existing vote
            await pool.query(
                'UPDATE review_helpfulness SET is_helpful = $1 WHERE review_id = $2 AND user_id = $3',
                [isHelpful, reviewId, userId]
            );
        } else {
            // Add new vote
            await pool.query(
                'INSERT INTO review_helpfulness (review_id, user_id, is_helpful) VALUES ($1, $2, $3)',
                [reviewId, userId, isHelpful]
            );
        }
        
        // Update helpful_votes count on the review
        const helpfulCount = await pool.query(
            'SELECT COUNT(*) as count FROM review_helpfulness WHERE review_id = $1 AND is_helpful = true',
            [reviewId]
        );
        
        await pool.query(
            'UPDATE product_reviews SET helpful_votes = $1 WHERE id = $2',
            [helpfulCount.rows[0].count, reviewId]
        );
        
        res.json({
            success: true,
            message: 'Vote recorded successfully',
            helpfulVotes: parseInt(helpfulCount.rows[0].count)
        });
    } catch (error) {
        console.error('Error voting on review:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Get user's reviews
router.get('/user/my-reviews', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const page = parseInt(req.query.page) || 1;
        const limit = parseInt(req.query.limit) || 10;
        const offset = (page - 1) * limit;
        
        const result = await pool.query(`
            SELECT 
                pr.*,
                p.title as product_title,
                p.images as product_images
            FROM product_reviews pr
            JOIN products p ON pr.product_id = p.id
            WHERE pr.user_id = $1
            ORDER BY pr.created_at DESC
            LIMIT $2 OFFSET $3
        `, [userId, limit, offset]);
        
        const countResult = await pool.query(
            'SELECT COUNT(*) FROM product_reviews WHERE user_id = $1',
            [userId]
        );
        
        res.json({
            success: true,
            reviews: result.rows,
            pagination: {
                page,
                limit,
                totalReviews: parseInt(countResult.rows[0].count),
                totalPages: Math.ceil(countResult.rows[0].count / limit)
            }
        });
    } catch (error) {
        console.error('Error fetching user reviews:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

// Check if user can review a product
router.get('/can-review/:productId', auth, async (req, res) => {
    try {
        const userId = req.user.id;
        const productId = req.params.productId;
        
        // Check if user already reviewed
        const existingReview = await pool.query(
            'SELECT id FROM product_reviews WHERE product_id = $1 AND user_id = $2',
            [productId, userId]
        );
        
        if (existingReview.rows.length > 0) {
            return res.json({
                success: true,
                canReview: false,
                reason: 'already_reviewed'
            });
        }
        
        // Check if user purchased the product
        const purchaseResult = await pool.query(`
            SELECT oi.id FROM order_items oi
            JOIN orders o ON oi.order_id = o.id
            WHERE o.user_id = $1 AND oi.product_id = $2 AND o.status IN ('delivered', 'confirmed')
        `, [userId, productId]);
        
        res.json({
            success: true,
            canReview: true,
            hasPurchased: purchaseResult.rows.length > 0
        });
    } catch (error) {
        console.error('Error checking review eligibility:', error);
        res.status(500).json({ success: false, message: 'Server error' });
    }
});

module.exports = router;