import React, { useState, useEffect, useRef, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import api from '../services/api';

const NotificationBell = () => {
  const [notifications, setNotifications] = useState([]);
  const [unreadCount, setUnreadCount] = useState(0);
  const [showDropdown, setShowDropdown] = useState(false);
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();
  const lastNotificationIdRef = useRef(null); // Track last seen notification ID
  const audioRef = useRef(null);
  const isInitialLoadRef = useRef(true);

  const requestNotificationPermission = async () => {
    if ('Notification' in window && Notification.permission === 'default') {
      const permission = await Notification.requestPermission();
      console.log('Notification permission:', permission);
    }
  };

  const playNotificationSound = useCallback(() => {
    try {
      console.log('🔊 Playing notification sound...');
      if (audioRef.current) {
        audioRef.current();
        console.log('✅ Sound played successfully');
      } else {
        console.log('❌ Audio ref is null');
      }
    } catch (err) {
      console.log('❌ Audio play failed:', err);
    }
  }, []);

  const showBrowserNotification = useCallback((title, body) => {
    console.log('🔔 Showing browser notification:', title, body);
    if ('Notification' in window && Notification.permission === 'granted') {
      const notification = new Notification(title, {
        body: body,
        icon: '/favicon.ico',
        badge: '/favicon.ico',
        tag: 'kupi-mk-notification',
        requireInteraction: false
      });

      notification.onclick = () => {
        window.focus();
        notification.close();
      };

      // Auto close after 5 seconds
      setTimeout(() => notification.close(), 5000);
      console.log('✅ Browser notification shown');
    } else {
      console.log('❌ Notification permission not granted or not supported');
    }
  }, []);

  const fetchUnreadCount = useCallback(async () => {
    try {
      // Fetch all unread notifications (not just count)
      const response = await api.get('/notifications?unread=true');
      const unreadNotifications = response.data;
      const newCount = unreadNotifications.length;
      
      console.log('📊 Unread notifications:', newCount, 'Last ID:', lastNotificationIdRef.current, 'IsInitial:', isInitialLoadRef.current);
      
      // Skip notification on initial load
      if (isInitialLoadRef.current) {
        console.log('⏭️ Skipping initial load');
        isInitialLoadRef.current = false;
        if (unreadNotifications.length > 0) {
          lastNotificationIdRef.current = unreadNotifications[0].id;
        }
        setUnreadCount(newCount);
        return;
      }
      
      // Check if there's a new notification (by comparing the latest notification ID)
      if (unreadNotifications.length > 0) {
        const latestNotification = unreadNotifications[0];
        
        if (lastNotificationIdRef.current !== null && latestNotification.id !== lastNotificationIdRef.current) {
          console.log('🆕 New notification detected! ID:', latestNotification.id, 'Previous ID:', lastNotificationIdRef.current);
          playNotificationSound();
          showBrowserNotification(
            latestNotification.title,
            latestNotification.message
          );
        }
        
        lastNotificationIdRef.current = latestNotification.id;
      }
      
      setUnreadCount(newCount);
    } catch (error) {
      console.error('Error fetching unread notifications:', error);
    }
  }, [playNotificationSound, showBrowserNotification]);

  useEffect(() => {
    console.log('🚀 NotificationBell mounted');
    
    // Request notification permission on mount
    requestNotificationPermission();
    
    // Create audio context and oscillator for notification sound (better browser compatibility)
    const createNotificationSound = () => {
      const audioContext = new (window.AudioContext || window.webkitAudioContext)();
      return () => {
        const oscillator = audioContext.createOscillator();
        const gainNode = audioContext.createGain();
        
        oscillator.connect(gainNode);
        gainNode.connect(audioContext.destination);
        
        oscillator.frequency.value = 800; // Frequency in Hz
        oscillator.type = 'sine';
        
        gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
        gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);
        
        oscillator.start(audioContext.currentTime);
        oscillator.stop(audioContext.currentTime + 0.5);
      };
    };
    
    try {
      audioRef.current = createNotificationSound();
      console.log('✅ Audio context created');
    } catch (err) {
      console.log('❌ Audio context not supported:', err);
    }
    
    fetchUnreadCount();
    // Poll for new notifications every 30 seconds
    const interval = setInterval(fetchUnreadCount, 30000);
    return () => {
      console.log('🛑 NotificationBell unmounted');
      clearInterval(interval);
    };
  }, [fetchUnreadCount]);

  const fetchNotifications = async () => {
    try {
      setLoading(true);
      const response = await api.get('/notifications');
      setNotifications(response.data);
    } catch (error) {
      console.error('Error fetching notifications:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleBellClick = () => {
    setShowDropdown(!showDropdown);
    if (!showDropdown) {
      fetchNotifications();
    }
  };

  const handleNotificationClick = async (notification) => {
    try {
      // Mark as read
      await api.patch(`/notifications/${notification.id}/read`);
      
      // Update local state
      setNotifications(notifications.map(n => 
        n.id === notification.id ? { ...n, is_read: true } : n
      ));
      setUnreadCount(prev => Math.max(0, prev - 1));
      
      // Navigate to order details if it's an order notification
      if (notification.order_id) {
        navigate(`/orders/${notification.order_id}`);
      }
      
      setShowDropdown(false);
    } catch (error) {
      console.error('Error marking notification as read:', error);
    }
  };

  const handleMarkAllRead = async () => {
    try {
      await api.patch('/notifications/read-all');
      setNotifications(notifications.map(n => ({ ...n, is_read: true })));
      setUnreadCount(0);
    } catch (error) {
      console.error('Error marking all as read:', error);
    }
  };

  const handleDelete = async (e, notificationId) => {
    e.stopPropagation();
    try {
      await api.delete(`/notifications/${notificationId}`);
      setNotifications(notifications.filter(n => n.id !== notificationId));
      fetchUnreadCount();
    } catch (error) {
      console.error('Error deleting notification:', error);
    }
  };

  const formatTimeAgo = (dateString) => {
    const date = new Date(dateString);
    const now = new Date();
    const seconds = Math.floor((now - date) / 1000);

    if (seconds < 60) return 'пред неколку секунди';
    if (seconds < 3600) return `пред ${Math.floor(seconds / 60)} минути`;
    if (seconds < 86400) return `пред ${Math.floor(seconds / 3600)} часа`;
    if (seconds < 604800) return `пред ${Math.floor(seconds / 86400)} дена`;
    return date.toLocaleDateString('mk-MK');
  };

  return (
    <div className="relative">
      {/* Bell Icon */}
      <button
        onClick={handleBellClick}
        className="relative p-2 text-gray-600 hover:text-gray-900 focus:outline-none"
      >
        <svg
          className="w-6 h-6"
          fill="none"
          stroke="currentColor"
          viewBox="0 0 24 24"
        >
          <path
            strokeLinecap="round"
            strokeLinejoin="round"
            strokeWidth={2}
            d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9"
          />
        </svg>
        {unreadCount > 0 && (
          <span className="absolute top-0 right-0 inline-flex items-center justify-center px-2 py-1 text-xs font-bold leading-none text-white transform translate-x-1/2 -translate-y-1/2 bg-red-600 rounded-full">
            {unreadCount > 9 ? '9+' : unreadCount}
          </span>
        )}
      </button>

      {/* Dropdown */}
      {showDropdown && (
        <>
          <div
            className="fixed inset-0 z-10"
            onClick={() => setShowDropdown(false)}
          ></div>
          <div className="absolute right-0 mt-2 w-80 bg-white rounded-lg shadow-lg z-20 max-h-96 overflow-y-auto">
            {/* Header */}
            <div className="p-4 border-b border-gray-200 flex justify-between items-center">
              <h3 className="text-lg font-semibold text-gray-900">
                Известувања
              </h3>
              {unreadCount > 0 && (
                <button
                  onClick={handleMarkAllRead}
                  className="text-sm text-primary-600 hover:text-primary-700"
                >
                  Označи сè како прочитано
                </button>
              )}
            </div>

            {/* Notifications List */}
            <div className="divide-y divide-gray-200">
              {loading ? (
                <div className="p-4 text-center text-gray-500">
                  Се вчитува...
                </div>
              ) : notifications.length === 0 ? (
                <div className="p-4 text-center text-gray-500">
                  Немате известувања
                </div>
              ) : (
                notifications.map((notification) => (
                  <div
                    key={notification.id}
                    onClick={() => handleNotificationClick(notification)}
                    className={`p-4 hover:bg-gray-50 cursor-pointer transition-colors ${
                      !notification.is_read ? 'bg-blue-50' : ''
                    }`}
                  >
                    <div className="flex justify-between items-start">
                      <div className="flex-1">
                        <p className="text-sm font-medium text-gray-900">
                          {notification.title}
                        </p>
                        <p className="text-sm text-gray-600 mt-1">
                          {notification.message}
                        </p>
                        <p className="text-xs text-gray-400 mt-2">
                          {formatTimeAgo(notification.created_at)}
                        </p>
                      </div>
                      <button
                        onClick={(e) => handleDelete(e, notification.id)}
                        className="ml-2 text-gray-400 hover:text-red-600"
                      >
                        <svg
                          className="w-4 h-4"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M6 18L18 6M6 6l12 12"
                          />
                        </svg>
                      </button>
                    </div>
                    {!notification.is_read && (
                      <div className="absolute left-2 top-1/2 transform -translate-y-1/2 w-2 h-2 bg-blue-600 rounded-full"></div>
                    )}
                  </div>
                ))
              )}
            </div>

            {/* Footer */}
            {notifications.length > 0 && (
              <div className="p-2 border-t border-gray-200">
                <button
                  onClick={() => {
                    navigate('/notifications');
                    setShowDropdown(false);
                  }}
                  className="w-full text-center text-sm text-primary-600 hover:text-primary-700 py-2"
                >
                  Види сè
                </button>
              </div>
            )}
          </div>
        </>
      )}
    </div>
  );
};

export default NotificationBell;
