const express = require('express');
const { pool } = require('../config/database');
const { auth } = require('../middleware/auth');

const router = express.Router();

// Get all favorites for logged-in user
router.get('/', auth, async (req, res) => {
  try {
    const userId = req.user.userId;

    const favorites = await pool.query(
      `SELECT 
        f.id as favorite_id,
        f.created_at as favorited_at,
        p.id,
        p.title,
        p.description,
        p.price,
        p.images,
        p.stock_quantity,
        p.location,
        p.is_active,
        c.name as category_name,
        u.username as seller_username
       FROM favorites f
       JOIN products p ON f.product_id = p.id
       LEFT JOIN categories c ON p.category_id = c.id
       LEFT JOIN users u ON p.seller_id = u.id
       WHERE f.user_id = $1
       ORDER BY f.created_at DESC`,
      [userId]
    );

    res.json(favorites.rows);
  } catch (error) {
    console.error('Error fetching favorites:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Check if product is favorited
router.get('/check/:productId', auth, async (req, res) => {
  try {
    const userId = req.user.userId;
    const productId = req.params.productId;

    const result = await pool.query(
      'SELECT id FROM favorites WHERE user_id = $1 AND product_id = $2',
      [userId, productId]
    );

    res.json({ isFavorited: result.rows.length > 0 });
  } catch (error) {
    console.error('Error checking favorite:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Add product to favorites
router.post('/:productId', auth, async (req, res) => {
  try {
    const userId = req.user.userId;
    const productId = req.params.productId;

    // Check if product exists
    const productCheck = await pool.query(
      'SELECT id FROM products WHERE id = $1 AND is_active = true',
      [productId]
    );

    if (productCheck.rows.length === 0) {
      return res.status(404).json({ message: 'Product not found' });
    }

    // Add to favorites (ignore if already exists due to UNIQUE constraint)
    await pool.query(
      'INSERT INTO favorites (user_id, product_id) VALUES ($1, $2) ON CONFLICT (user_id, product_id) DO NOTHING',
      [userId, productId]
    );

    res.json({ success: true, message: 'Added to favorites' });
  } catch (error) {
    console.error('Error adding to favorites:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Remove product from favorites
router.delete('/:productId', auth, async (req, res) => {
  try {
    const userId = req.user.userId;
    const productId = req.params.productId;

    const result = await pool.query(
      'DELETE FROM favorites WHERE user_id = $1 AND product_id = $2 RETURNING *',
      [userId, productId]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'Favorite not found' });
    }

    res.json({ success: true, message: 'Removed from favorites' });
  } catch (error) {
    console.error('Error removing from favorites:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

module.exports = router;
