import React, { useState, useEffect } from 'react';
import { useAuth } from '../context/AuthContext';
import { useNavigate } from 'react-router-dom';
import api from '../services/api';

const Profile = () => {
  const { user, setUser } = useAuth();
  const navigate = useNavigate();
  const [userProducts, setUserProducts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [editForm, setEditForm] = useState({
    username: '',
    first_name: '',
    last_name: '',
    phone: '',
    address: ''
  });
  const [isUpdating, setIsUpdating] = useState(false);

  const fetchUserProducts = async () => {
    try {
      const response = await api.get('/users/my-products');
      setUserProducts(response.data);
    } catch (error) {
      console.error('Error fetching user products:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleEditProduct = (productId) => {
    navigate(`/edit-product/${productId}`);
  };

  const handleDeleteProduct = async (productId, productTitle) => {
    if (window.confirm(`Дали сте сигурни дека сакате да го деактивирате производот "${productTitle}"? (може да се реактивира подоцна)`)) {
      try {
        await api.delete(`/products/${productId}`);
        // Refresh the products list
        fetchUserProducts();
        alert('Производот е успешно деактивиран!');
      } catch (error) {
        console.error('Error deactivating product:', error);
        alert('Грешка при деактивирање на производот');
      }
    }
  };

  const handleReactivateProduct = async (productId, productTitle) => {
    if (window.confirm(`Дали сте сигурни дека сакате да го реактивирате производот "${productTitle}"?`)) {
      try {
        await api.patch(`/products/${productId}/reactivate`);
        // Refresh the products list
        fetchUserProducts();
        alert('Производот е успешно реактивиран!');
      } catch (error) {
        console.error('Error reactivating product:', error);
        alert('Грешка при реактивирање на производот');
      }
    }
  };

  const handlePermanentDelete = async (productId, productTitle) => {
    if (window.confirm(`⚠️ ПРЕДУПРЕДУВАЊЕ: Дали сте сигурни дека сакате ТРАЈНО да го избришете производот "${productTitle}"? Оваа акција НЕ МОЖЕ да се врати!`)) {
      if (window.confirm('Последно предупредување: Производот ќе биде трајно избришан. Продолжи?')) {
        try {
          await api.delete(`/products/${productId}/permanent`);
          // Refresh the products list
          fetchUserProducts();
          alert('Производот е трајно избришан!');
        } catch (error) {
          console.error('Error permanently deleting product:', error);
          alert('Грешка при трајно бришење на производот');
        }
      }
    }
  };

  const handleEditProfile = () => {
    setEditForm({
      username: user?.username || '',
      first_name: user?.first_name || '',
      last_name: user?.last_name || '',
      phone: user?.phone || '',
      address: user?.address || ''
    });
    setIsEditModalOpen(true);
  };

  const handleUpdateProfile = async (e) => {
    e.preventDefault();
    setIsUpdating(true);

    try {
      const response = await api.put('/users/profile', editForm);
      setUser(response.data);
      setIsEditModalOpen(false);
      alert('Профилот е успешно ажуриран!');
    } catch (error) {
      console.error('Error updating profile:', error);
      if (error.response?.data?.message === 'Username is already taken') {
        alert('Корисничкото име е веќе зафатено. Обидете се со друго.');
      } else {
        alert('Грешка при ажурирање на профилот');
      }
    } finally {
      setIsUpdating(false);
    }
  };

  const handleBecomeSeller = async () => {
    if (window.confirm('Дали сакате да станете продавач? Ова ќе ви овозможи да објавувате производи.')) {
      try {
        const response = await api.patch('/users/become-seller');
        setUser(response.data);
        alert('Успешно станавте продавач!');
      } catch (error) {
        console.error('Error becoming seller:', error);
        alert('Грешка при станување продавач');
      }
    }
  };

  useEffect(() => {
    fetchUserProducts();
  }, []);

  if (loading) {
    return (
      <div className="flex justify-center items-center min-h-screen">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-primary-600"></div>
      </div>
    );
  }

  return (
    <div className="max-w-4xl mx-auto">
      {/* Profile Header */}
      <div className="bg-white rounded-lg shadow-lg p-6 mb-8">
        <div className="flex items-center space-x-6">
          <div className="w-20 h-20 bg-primary-600 rounded-full flex items-center justify-center text-white text-2xl font-bold">
            {user?.first_name?.charAt(0)}{user?.last_name?.charAt(0)}
          </div>
          
          <div className="flex-1">
            <h1 className="text-3xl font-bold text-gray-900">
              {user?.first_name} {user?.last_name}
            </h1>
            <p className="text-gray-600">@{user?.username}</p>
            <p className="text-gray-600">{user?.email}</p>
            
            {user?.is_seller && (
              <span className="inline-block mt-2 bg-green-100 text-green-800 px-3 py-1 rounded-full text-sm">
                Веrifициран продавач
              </span>
            )}
          </div>
          
          <div className="flex space-x-3">
            <button 
              onClick={handleEditProfile}
              className="btn-secondary"
            >
              Уреди профил
            </button>
            
            {!user?.is_seller && (
              <button 
                onClick={handleBecomeSeller}
                className="btn-primary"
              >
                Стани продавач
              </button>
            )}
          </div>
        </div>
        
        {user?.phone && (
          <div className="mt-4 text-gray-600">
            📞 {user.phone}
          </div>
        )}
        
        {user?.address && (
          <div className="mt-2 text-gray-600">
            📍 {user.address}
          </div>
        )}
      </div>

      {/* User's Products */}
      <div className="bg-white rounded-lg shadow-lg p-6">
        <div className="flex justify-between items-center mb-6">
          <h2 className="text-2xl font-bold text-gray-900">
            Мои производи ({userProducts.length})
          </h2>
          <button 
            onClick={() => window.location.href = '/create-product'}
            className="btn-primary"
          >
            + Додај нов производ
          </button>
        </div>

        {userProducts.length === 0 ? (
          <div className="text-center py-12">
            <div className="text-gray-500 text-lg mb-4">
              Немате објавени производи
            </div>
            <p className="text-gray-400 mb-6">
              Започнете со објавување на вашиот прв производ
            </p>
            <button 
              onClick={() => window.location.href = '/create-product'}
              className="btn-primary"
            >
              Додај производ
            </button>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {userProducts.map(product => (
              <div key={product.id} className="border border-gray-200 rounded-lg overflow-hidden hover:shadow-md transition-shadow">
                <div className="aspect-w-16 aspect-h-9 bg-gray-200">
                  <img
                    src={product.images && product.images.length > 0 
                      ? `http://localhost:5001${product.images[0]}` 
                      : '/placeholder-image.jpg'
                    }
                    alt={product.title}
                    className="w-full h-48 object-cover"
                    onError={(e) => {
                      e.target.src = '/placeholder-image.jpg';
                    }}
                  />
                </div>
                
                <div className="p-4">
                  <h3 className="text-lg font-semibold text-gray-900 mb-2 line-clamp-2">
                    {product.title}
                  </h3>
                  
                  <p className="text-gray-600 text-sm mb-3 line-clamp-2">
                    {product.description}
                  </p>
                  
                  <div className="flex justify-between items-center mb-3">
                    <span className="text-xl font-bold text-primary-600">
                      {parseFloat(product.price).toLocaleString('mk-MK')} ден.
                    </span>
                    
                    <span className={`px-2 py-1 rounded-full text-xs ${
                      product.is_active 
                        ? 'bg-green-100 text-green-800' 
                        : 'bg-red-100 text-red-800'
                    }`}>
                      {product.is_active ? 'Активен' : 'Неактивен'}
                    </span>
                  </div>
                  
                  <div className="flex justify-between items-center text-sm text-gray-500 mb-4">
                    <span>{product.category_name}</span>
                    <span>
                      {new Date(product.created_at).toLocaleDateString('mk-MK')}
                    </span>
                  </div>
                  
                  <div className="flex space-x-2">
                    <button 
                      onClick={() => navigate(`/product/${product.id}`)}
                      className="flex-1 text-primary-600 border border-primary-600 px-2 py-1.5 rounded text-xs hover:bg-primary-50 transition-colors"
                    >
                      👁️
                    </button>
                    <button 
                      onClick={() => handleEditProduct(product.id)}
                      className="flex-1 bg-blue-600 text-white px-2 py-1.5 rounded text-xs hover:bg-blue-700 transition-colors"
                    >
                      ✏️
                    </button>
                    {product.is_active ? (
                      <button 
                        onClick={() => handleDeleteProduct(product.id, product.title)}
                        className="flex-1 bg-red-600 text-white px-2 py-1.5 rounded text-xs hover:bg-red-700 transition-colors"
                      >
                        🚫
                      </button>
                    ) : (
                      <>
                        <button 
                          onClick={() => handleReactivateProduct(product.id, product.title)}
                          className="flex-1 bg-green-600 text-white px-2 py-1.5 rounded text-xs hover:bg-green-700 transition-colors"
                        >
                          ✅
                        </button>
                        <button 
                          onClick={() => handlePermanentDelete(product.id, product.title)}
                          className="flex-1 bg-red-800 text-white px-2 py-1.5 rounded text-xs hover:bg-red-900 transition-colors"
                        >
                          🗑️
                        </button>
                      </>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>

      {/* Edit Profile Modal */}
      {isEditModalOpen && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
            <h3 className="text-xl font-bold text-gray-900 mb-4">Уреди профил</h3>
            
            <form onSubmit={handleUpdateProfile} className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Корисничко име (@име)
                </label>
                <input
                  type="text"
                  value={editForm.username}
                  onChange={(e) => setEditForm({ ...editForm, username: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  required
                  placeholder="korisnicko_ime"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Име
                </label>
                <input
                  type="text"
                  value={editForm.first_name}
                  onChange={(e) => setEditForm({ ...editForm, first_name: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Презиме
                </label>
                <input
                  type="text"
                  value={editForm.last_name}
                  onChange={(e) => setEditForm({ ...editForm, last_name: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Телефон
                </label>
                <input
                  type="tel"
                  value={editForm.phone}
                  onChange={(e) => setEditForm({ ...editForm, phone: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  placeholder="070 123 456"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Адреса
                </label>
                <input
                  type="text"
                  value={editForm.address}
                  onChange={(e) => setEditForm({ ...editForm, address: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  placeholder="Град, улица..."
                />
              </div>
              
              <div className="flex space-x-3 pt-4">
                <button
                  type="button"
                  onClick={() => setIsEditModalOpen(false)}
                  className="flex-1 px-4 py-2 text-gray-600 border border-gray-300 rounded-md hover:bg-gray-50 transition-colors"
                  disabled={isUpdating}
                >
                  Откажи
                </button>
                <button
                  type="submit"
                  className="flex-1 bg-primary-600 text-white px-4 py-2 rounded-md hover:bg-primary-700 transition-colors disabled:opacity-50"
                  disabled={isUpdating}
                >
                  {isUpdating ? 'Се зачувува...' : 'Зачувај'}
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default Profile;
