const { Pool } = require('pg');

const pool = new Pool({
  host: process.env.DB_HOST || 'localhost',
  port: process.env.DB_PORT || 5432,
  database: process.env.DB_NAME || 'kupi_mk',
  user: process.env.DB_USER,
  password: process.env.DB_PASSWORD,
});

const connectDB = async () => {
  try {
    await pool.connect();
    console.log('PostgreSQL Connected');
    
    // Create tables if they don't exist
    await createTables();
  } catch (err) {
    console.error('Database connection error:', err.message);
    process.exit(1);
  }
};

const createTables = async () => {
  const client = await pool.connect();
  try {
    // Users table
    await client.query(`
      CREATE TABLE IF NOT EXISTS users (
        id SERIAL PRIMARY KEY,
        username VARCHAR(50) UNIQUE NOT NULL,
        email VARCHAR(100) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        first_name VARCHAR(50) NOT NULL,
        last_name VARCHAR(50) NOT NULL,
        phone VARCHAR(20),
        address TEXT,
        is_seller BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Categories table
    await client.query(`
      CREATE TABLE IF NOT EXISTS categories (
        id SERIAL PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        description TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Products table
    await client.query(`
      CREATE TABLE IF NOT EXISTS products (
        id SERIAL PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        description TEXT NOT NULL,
        price DECIMAL(10,2) NOT NULL,
        category_id INTEGER REFERENCES categories(id),
        seller_id INTEGER REFERENCES users(id),
        images TEXT[], -- Array of image URLs
        stock_quantity INTEGER DEFAULT 0,
        is_active BOOLEAN DEFAULT TRUE,
        location VARCHAR(100),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    // Insert default categories (only if they don't exist)
    const existingCategories = await client.query('SELECT COUNT(*) FROM categories');
    if (parseInt(existingCategories.rows[0].count) === 0) {
      await client.query(`
        INSERT INTO categories (name, description) 
        VALUES 
          ('Електроника', 'Телефони, лаптопи, компјутери'),
          ('Облека', 'Машка, женска и детска облека'),
          ('Дом и градина', 'Мебел, декорации, алатки'),
          ('Спорт и рекреација', 'Спортска опрема и додатоци'),
          ('Книги и музика', 'Книги, CD-а, винили'),
          ('Автомобили', 'Возила и авто делови'),
          ('Храна и пијалоци', 'Локални производи'),
          ('Убавина и здравје', 'Козметика и здравствени производи')
      `);
      console.log('Default categories inserted');
    } else {
      console.log('Categories already exist, skipping insertion');
    }

    console.log('Database tables created successfully');
  } catch (err) {
    console.error('Error creating tables:', err);
  } finally {
    client.release();
  }
};

module.exports = { pool, connectDB };
